<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>VAPI Manager - Validated & Stable</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
            position: relative;
            overflow-x: hidden;
        }
        
        /* Animated background particles */
        body::before {
            content: '';
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: 
                radial-gradient(circle at 20% 50%, rgba(255,255,255,0.1) 0%, transparent 50%),
                radial-gradient(circle at 80% 80%, rgba(255,255,255,0.1) 0%, transparent 50%),
                radial-gradient(circle at 40% 20%, rgba(255,255,255,0.1) 0%, transparent 50%);
            animation: float 20s ease-in-out infinite;
            pointer-events: none;
        }
        
        @keyframes float {
            0%, 100% { transform: translateY(0px); }
            50% { transform: translateY(-20px); }
        }
        
        .container { 
            max-width: 1400px; 
            margin: 0 auto;
            position: relative;
            z-index: 1;
        }
        
        .header {
            background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%);
            padding: 40px;
            border-radius: 20px;
            text-align: center;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3);
            margin-bottom: 30px;
            position: relative;
            overflow: hidden;
        }
        
        .header::before {
            content: '';
            position: absolute;
            top: -50%;
            left: -50%;
            width: 200%;
            height: 200%;
            background: linear-gradient(45deg, transparent, rgba(102, 126, 234, 0.1), transparent);
            animation: shimmer 3s infinite;
        }
        
        @keyframes shimmer {
            0% { transform: translateX(-100%) translateY(-100%) rotate(45deg); }
            100% { transform: translateX(100%) translateY(100%) rotate(45deg); }
        }
        
        .header h1 {
            color: #667eea;
            font-size: 2.8em;
            margin-bottom: 10px;
            font-weight: 800;
            position: relative;
            text-shadow: 2px 2px 4px rgba(0,0,0,0.1);
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }
        
        .header p {
            color: #666;
            font-size: 1.1em;
            position: relative;
            animation: fadeIn 1s ease;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(10px); }
            to { opacity: 1; transform: translateY(0); }
        }
        
        .btn {
            padding: 14px 28px;
            border: none;
            border-radius: 12px;
            cursor: pointer;
            font-size: 16px;
            font-weight: bold;
            transition: all 0.3s ease;
            margin: 5px;
            position: relative;
            overflow: hidden;
        }
        
        .btn::before {
            content: '';
            position: absolute;
            top: 50%;
            left: 50%;
            width: 0;
            height: 0;
            border-radius: 50%;
            background: rgba(255,255,255,0.3);
            transform: translate(-50%, -50%);
            transition: width 0.6s, height 0.6s;
        }
        
        .btn:hover::before {
            width: 300px;
            height: 300px;
        }
        
        .btn-primary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            box-shadow: 0 4px 15px rgba(102, 126, 234, 0.4);
            position: relative;
            z-index: 1;
        }
        
        .btn-primary:hover {
            transform: translateY(-3px) scale(1.05);
            box-shadow: 0 8px 25px rgba(102, 126, 234, 0.5);
        }
        
        .btn-primary:active {
            transform: translateY(-1px) scale(1.02);
        }
        
        .btn-danger {
            background: linear-gradient(135deg, #e74c3c 0%, #c0392b 100%);
            color: white;
            box-shadow: 0 4px 15px rgba(231, 76, 60, 0.4);
        }
        
        .btn-danger:hover {
            transform: translateY(-3px);
            box-shadow: 0 8px 25px rgba(231, 76, 60, 0.5);
        }
        
        .loading {
            text-align: center;
            padding: 60px;
        }
        
        .spinner {
            border: 5px solid rgba(255,255,255,0.3);
            border-top: 5px solid #667eea;
            border-radius: 50%;
            width: 60px;
            height: 60px;
            animation: spin 1s cubic-bezier(0.68, -0.55, 0.265, 1.55) infinite;
            margin: 20px auto;
            box-shadow: 0 0 20px rgba(102, 126, 234, 0.3);
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        .message {
            padding: 18px 24px 18px 50px;
            border-radius: 12px;
            margin: 10px 0;
            max-width: 600px;
            animation: slideIn 0.3s ease;
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            position: relative;
        }
        
        .message::before {
            content: '';
            position: absolute;
            left: 18px;
            top: 50%;
            transform: translateY(-50%);
            width: 24px;
            height: 24px;
            background-size: contain;
            background-repeat: no-repeat;
            animation: iconBounce 0.5s ease;
        }
        
        @keyframes iconBounce {
            0%, 100% { transform: translateY(-50%) scale(1); }
            50% { transform: translateY(-50%) scale(1.2); }
        }
        
        .message.error::before {
            content: '❌';
            font-size: 20px;
        }
        
        .message.success::before {
            content: '✅';
            font-size: 20px;
        }
        
        @keyframes slideIn {
            from {
                transform: translateY(-20px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
        
        .message.error {
            background: linear-gradient(135deg, #f8d7da 0%, #f5c6cb 100%);
            color: #721c24;
            border-left: 4px solid #e74c3c;
        }
        
        .message.success {
            background: linear-gradient(135deg, #d4edda 0%, #c3e6cb 100%);
            color: #155724;
            border-left: 4px solid #28a745;
        }
        
        .content {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            border-radius: 20px;
            padding: 35px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3);
        }
        
        .toolbar {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 25px;
            flex-wrap: wrap;
            gap: 12px;
        }
        
        .search-box {
            padding: 12px 18px 12px 45px;
            border: 2px solid #667eea;
            border-radius: 12px;
            font-size: 16px;
            flex: 1;
            max-width: 400px;
            transition: all 0.3s ease;
            background: white url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="%23667eea" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="11" cy="11" r="8"></circle><path d="m21 21-4.35-4.35"></path></svg>') no-repeat 15px center;
        }
        
        .search-box:focus {
            outline: none;
            border-color: #764ba2;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
            transform: translateY(-2px);
            background-position: 12px center;
        }
        
        .assistants-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
            gap: 25px;
            margin-top: 25px;
        }
        
        .assistant-card {
            background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%);
            border-radius: 16px;
            padding: 25px;
            border: 2px solid transparent;
            transition: all 0.4s cubic-bezier(0.175, 0.885, 0.32, 1.275);
            position: relative;
            overflow: hidden;
            animation: cardAppear 0.5s ease forwards;
            opacity: 0;
            transform: translateY(20px);
        }
        
        @keyframes cardAppear {
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        .assistant-card:nth-child(1) { animation-delay: 0.1s; }
        .assistant-card:nth-child(2) { animation-delay: 0.2s; }
        .assistant-card:nth-child(3) { animation-delay: 0.3s; }
        .assistant-card:nth-child(4) { animation-delay: 0.4s; }
        .assistant-card:nth-child(5) { animation-delay: 0.5s; }
        .assistant-card:nth-child(6) { animation-delay: 0.6s; }
        
        .assistant-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 4px;
            background: linear-gradient(90deg, #667eea 0%, #764ba2 100%);
            transform: scaleX(0);
            transition: transform 0.3s ease;
        }
        
        .assistant-card:hover::before {
            transform: scaleX(1);
        }
        
        .assistant-card:hover {
            transform: translateY(-8px) scale(1.02);
            box-shadow: 0 15px 40px rgba(102, 126, 234, 0.25);
            border-color: #667eea;
        }
        
        .assistant-card h3 {
            color: #2c3e50;
            margin-bottom: 20px;
            padding-bottom: 12px;
            border-bottom: 3px solid #667eea;
            font-size: 1.4em;
            font-weight: 700;
        }
        
        .assistant-info {
            margin: 18px 0;
        }
        
        .assistant-info label {
            font-weight: 600;
            color: #555;
            display: block;
            margin-bottom: 8px;
            font-size: 13px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .assistant-info input,
        .assistant-info textarea,
        .assistant-info select {
            width: 100%;
            padding: 12px;
            border: 2px solid #e9ecef;
            border-radius: 10px;
            font-size: 14px;
            background: white;
            transition: all 0.3s ease;
        }
        
        .assistant-info input:focus,
        .assistant-info textarea:focus,
        .assistant-info select:focus {
            outline: none;
            border-color: #667eea;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }
        
        .assistant-info textarea {
            min-height: 60px;
            resize: vertical;
        }
        
        .card-actions {
            display: flex;
            gap: 12px;
            margin-top: 20px;
        }
        
        .modal {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            overflow: auto;
            background-color: rgba(0,0,0,0.6);
            backdrop-filter: blur(5px);
            padding: 20px;
            animation: fadeIn 0.3s ease;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }
        
        .modal-content {
            background-color: #fefefe;
            margin: 20px auto;
            padding: 0;
            border: none;
            border-radius: 20px;
            max-width: 900px;
            max-height: 85vh;
            overflow: hidden;
            display: flex;
            flex-direction: column;
            box-shadow: 0 30px 90px rgba(0,0,0,0.4);
            animation: slideUp 0.4s cubic-bezier(0.175, 0.885, 0.32, 1.275);
        }
        
        @keyframes slideUp {
            from {
                transform: translateY(50px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
        
        .modal-header {
            padding: 25px 35px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            display: flex;
            justify-content: space-between;
            align-items: center;
            position: relative;
            overflow: hidden;
        }
        
        .modal-header::before {
            content: '';
            position: absolute;
            top: -50%;
            right: -50%;
            width: 200%;
            height: 200%;
            background: radial-gradient(circle, rgba(255,255,255,0.1) 0%, transparent 70%);
            animation: pulse 4s ease-in-out infinite;
        }
        
        @keyframes pulse {
            0%, 100% { transform: scale(1); opacity: 0.5; }
            50% { transform: scale(1.1); opacity: 0.8; }
        }
        
        .modal-header h2 {
            margin: 0;
            font-size: 1.6em;
            font-weight: 700;
            position: relative;
            text-shadow: 2px 2px 4px rgba(0,0,0,0.2);
        }
        
        .modal-close {
            color: white;
            font-size: 32px;
            font-weight: bold;
            cursor: pointer;
            background: rgba(255,255,255,0.2);
            border: none;
            padding: 5px 12px;
            line-height: 1;
            border-radius: 8px;
            transition: all 0.3s ease;
            position: relative;
        }
        
        .modal-close:hover {
            background: rgba(255,255,255,0.3);
            transform: rotate(90deg);
        }
        
        .modal-body {
            padding: 35px;
            overflow-y: auto;
            flex: 1;
            background: linear-gradient(to bottom, #ffffff 0%, #f8f9fa 100%);
        }
        
        .modal-body::-webkit-scrollbar {
            width: 8px;
        }
        
        .modal-body::-webkit-scrollbar-track {
            background: #f1f1f1;
            border-radius: 10px;
        }
        
        .modal-body::-webkit-scrollbar-thumb {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border-radius: 10px;
        }
        
        .modal-footer {
            padding: 25px 35px;
            background: #f8f9fa;
            border-top: 2px solid #e9ecef;
            display: flex;
            gap: 12px;
            justify-content: center;
        }
        
        .form-section {
            background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%);
            padding: 25px;
            border-radius: 15px;
            margin: 25px 0;
            border-right: 5px solid #667eea;
            box-shadow: 0 4px 15px rgba(0,0,0,0.08);
            transition: all 0.3s ease;
        }
        
        .form-section:hover {
            transform: translateX(-5px);
            box-shadow: 0 6px 20px rgba(102, 126, 234, 0.15);
        }
        
        .form-section h3 {
            color: #667eea;
            margin-bottom: 20px;
            font-size: 1.4em;
            font-weight: 700;
            display: flex;
            align-items: center;
            gap: 10px;
            text-shadow: 0 0 20px rgba(102, 126, 234, 0.3);
        }
        
        .form-row {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 20px;
            margin-bottom: 20px;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group label {
            display: block;
            font-weight: 600;
            color: #555;
            margin-bottom: 8px;
            font-size: 14px;
        }
        
        .form-group input[type="text"],
        .form-group input[type="number"],
        .form-group textarea,
        .form-group select {
            width: 100%;
            padding: 12px 15px;
            border: 2px solid #e9ecef;
            border-radius: 10px;
            font-size: 14px;
            transition: all 0.3s ease;
            background: white;
        }
        
        .form-group input:focus,
        .form-group textarea:focus,
        .form-group select:focus {
            outline: none;
            border-color: #667eea;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
            transform: translateY(-2px);
        }
        
        .form-group textarea {
            min-height: 100px;
            resize: vertical;
            font-family: inherit;
        }
        
        .form-group input[type="checkbox"] {
            width: 20px;
            height: 20px;
            cursor: pointer;
            accent-color: #667eea;
        }
        
        .checkbox-group {
            display: flex;
            align-items: center;
            gap: 12px;
            padding: 12px;
            background: rgba(102, 126, 234, 0.05);
            border-radius: 10px;
            transition: all 0.3s ease;
        }
        
        .checkbox-group:hover {
            background: rgba(102, 126, 234, 0.1);
        }
        
        .checkbox-group input[type="checkbox"] {
            margin: 0;
        }
        
        .checkbox-group label {
            margin: 0;
            font-weight: 500;
            cursor: pointer;
        }
        
        #messageContainer {
            position: fixed;
            top: 20px;
            left: 50%;
            transform: translateX(-50%);
            z-index: 10000;
            min-width: 300px;
            max-width: 600px;
        }
        
        @media (max-width: 768px) {
            .form-row {
                grid-template-columns: 1fr;
            }
            
            .modal-content {
                max-width: 95%;
            }
            
            .header h1 {
                font-size: 2em;
            }
            
            .assistants-grid {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <div id="messageContainer"></div>
    
    <div class="container">
        <div class="header">
            <h1>🎙️ VAPI Manager - مُحقق ومستقر</h1>
            <p>إدارة المساعدين الصوتيين - النسخة المُحققة والموثوقة</p>
        </div>

        <div class="content">
            <div class="toolbar">
                <button class="btn btn-primary" onclick="openCreateModal()">➕ إنشاء مساعد جديد</button>
                <input type="text" id="search-assistants" class="search-box" placeholder="🔍 ابحث عن مساعد..." onkeyup="filterAssistants()" />
                <button class="btn btn-primary" onclick="loadAssistants()">🔄 تحديث</button>
            </div>
            
            <div id="assistants-container">
                <div class="loading">
                    <div class="spinner"></div>
                    <p>جاري تحميل المساعدين...</p>
                </div>
            </div>
        </div>
    </div>

    <!-- CREATE/EDIT ASSISTANT MODAL -->
    <div id="assistantModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h2 id="modalTitle">إنشاء مساعد جديد</h2>
                <button class="modal-close" onclick="closeModal()">&times;</button>
            </div>
            
            <div class="modal-body">
                <form id="assistantForm">
                    <input type="hidden" id="assistant_id" />
                    
                    <!-- BASIC INFO -->
                    <div class="form-section">
                        <h3>📋 المعلومات الأساسية</h3>
                        <div class="form-group">
                            <label>اسم المساعد: *</label>
                            <input type="text" id="name" required placeholder="مثال: مساعد خدمة العملاء" />
                        </div>
                        
                        <div class="form-group">
                            <label>الرسالة الأولى:</label>
                            <textarea id="firstMessage" rows="3" placeholder="مثال: مرحباً! كيف يمكنني مساعدتك اليوم؟"></textarea>
                        </div>
                        
                        <div class="form-group">
                            <label>الرسالة الأخيرة (End Call Message):</label>
                            <textarea id="endCallMessage" rows="2" placeholder="مثال: شكراً لك، وداعاً!"></textarea>
                        </div>
                    </div>

                    <!-- PROMPTS -->
                    <div class="form-section">
                        <h3>💬 System Prompt (التعليمات الأساسية)</h3>
                        
                        <div class="form-group">
                            <label>System Prompt: *</label>
                            <textarea id="systemPrompt" rows="8" required placeholder="أنت مساعد ذكي..."></textarea>
                        </div>
                    </div>

                    <!-- ANALYSIS PROMPTS -->
                    <div class="form-section">
                        <h3>📊 Analysis Prompts (تحليل المكالمات)</h3>
                        
                        <div class="form-group">
                            <label>Success Evaluation Prompt:</label>
                            <textarea id="successEvalPrompt" rows="4" placeholder="قيّم نجاح المكالمة..."></textarea>
                        </div>
                        
                        <div class="form-group">
                            <label>Summary Prompt:</label>
                            <textarea id="summaryPrompt" rows="4" placeholder="لخص المكالمة..."></textarea>
                        </div>
                    </div>
                </form>
            </div>
            
            <div class="modal-footer">
                <button type="button" class="btn btn-primary" onclick="saveAssistant(event)">💾 حفظ</button>
                <button type="button" class="btn btn-danger" onclick="closeModal()">❌ إلغاء</button>
            </div>
        </div>
    </div>

    <script>
        let assistants = [];
        
        function showMessage(text, type) {
            const container = document.getElementById('messageContainer');
            const message = document.createElement('div');
            message.className = `message ${type}`;
            message.textContent = text;
            container.appendChild(message);
            
            setTimeout(() => {
                message.remove();
            }, 5000);
        }
        
        function escapeHtml(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
        
        async function loadAssistants() {
            const container = document.getElementById('assistants-container');
            container.innerHTML = '<div class="loading"><div class="spinner"></div><p>جاري التحميل...</p></div>';
            
            try {
                const response = await fetch('vapi_api.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ action: 'list_assistants', limit: 100 })
                });
                
                const data = await response.json();
                
                if (data.error) {
                    container.innerHTML = `<div class="message error">خطأ: ${data.message || JSON.stringify(data)}</div>`;
                    return;
                }
                
                assistants = Array.isArray(data) ? data : [];
                
                if (assistants.length === 0) {
                    container.innerHTML = '<p style="text-align: center; padding: 40px; color: #666;">لا توجد مساعدين. قم بإنشاء مساعد جديد!</p>';
                    return;
                }
                
                displayAssistants(assistants);
                showMessage(`تم تحميل ${assistants.length} مساعد بنجاح`, 'success');
                
            } catch (error) {
                container.innerHTML = `<div class="message error">خطأ: ${error.message}</div>`;
            }
        }
        
        function displayAssistants(assistantsList) {
            const container = document.getElementById('assistants-container');
            container.innerHTML = `
                <div class="assistants-grid">
                    ${assistantsList.map(assistant => `
                        <div class="assistant-card" data-id="${assistant.id}" data-name="${escapeHtml(assistant.name || '').toLowerCase()}">
                            <h3>${escapeHtml(assistant.name || 'بدون اسم')}</h3>
                            
                            <div class="assistant-info">
                                <label>المعرف:</label>
                                <input type="text" value="${assistant.id}" readonly style="font-size: 11px;" />
                            </div>
                            
                            <div class="assistant-info">
                                <label>النموذج:</label>
                                <input type="text" value="${assistant.model?.model || 'N/A'}" readonly />
                            </div>
                            
                            <div class="assistant-info">
                                <label>الصوت:</label>
                                <input type="text" value="${assistant.voice?.voiceId || 'N/A'}" readonly />
                            </div>
                            
                            <div class="card-actions">
                                <button class="btn btn-primary" onclick="editAssistant('${assistant.id}')">✏️ تعديل</button>
                                <button class="btn btn-danger" onclick="confirmDelete('${assistant.id}', '${escapeHtml(assistant.name || 'هذا المساعد').replace(/'/g, "\\'")}')">🗑️ حذف</button>
                            </div>
                        </div>
                    `).join('')}
                </div>
            `;
        }
        
        function filterAssistants() {
            const searchTerm = document.getElementById('search-assistants').value.toLowerCase();
            const cards = document.querySelectorAll('.assistant-card');
            
            cards.forEach(card => {
                const name = card.getAttribute('data-name');
                card.style.display = name.includes(searchTerm) ? '' : 'none';
            });
        }
        
        function openCreateModal() {
            document.getElementById('modalTitle').textContent = 'إنشاء مساعد جديد';
            document.getElementById('assistantForm').reset();
            document.getElementById('assistant_id').value = '';
            document.getElementById('assistantModal').style.display = 'block';
        }
        
        function closeModal() {
            document.getElementById('assistantModal').style.display = 'none';
        }
        
        async function editAssistant(assistantId) {
            try {
                const response = await fetch('vapi_api.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ 
                        action: 'get_assistant', 
                        assistant_id: assistantId 
                    })
                });
                
                const assistant = await response.json();
                
                if (assistant.error) {
                    showMessage('خطأ: ' + (assistant.message || JSON.stringify(assistant)), 'error');
                    return;
                }
                
                // Fill form - SIMPLIFIED
                document.getElementById('modalTitle').textContent = 'تعديل المساعد';
                document.getElementById('assistant_id').value = assistant.id;
                document.getElementById('name').value = assistant.name || '';
                document.getElementById('firstMessage').value = assistant.firstMessage || '';
                document.getElementById('endCallMessage').value = assistant.endCallMessage || '';
                document.getElementById('systemPrompt').value = assistant.model?.messages?.[0]?.content || '';
                
                // Analysis prompts
                document.getElementById('successEvalPrompt').value = assistant.analysisPlan?.successEvaluationPlan?.messages?.[0]?.content || '';
                document.getElementById('summaryPrompt').value = assistant.analysisPlan?.summaryPlan?.messages?.[0]?.content || '';
                
                document.getElementById('assistantModal').style.display = 'block';
            } catch (error) {
                showMessage('خطأ: ' + error.message, 'error');
            }
        }
        
        async function saveAssistant(event) {
            if (event) event.preventDefault();
            
            const assistantId = document.getElementById('assistant_id').value;
            const isEdit = !!assistantId;
            
            if (isEdit) {
                // EDIT MODE - Update existing assistant
                try {
                    const getResponse = await fetch('vapi_api.php', {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/json' },
                        body: JSON.stringify({ 
                            action: 'get_assistant', 
                            assistant_id: assistantId 
                        })
                    });
                    
                    const currentAssistant = await getResponse.json();
                    
                    if (currentAssistant.error) {
                        showMessage('خطأ: ' + (currentAssistant.message || JSON.stringify(currentAssistant)), 'error');
                        return;
                    }
                    
                    // Build updates - preserve existing model config and only update messages
                    const updates = {
                        name: document.getElementById('name').value,
                        firstMessage: document.getElementById('firstMessage').value || undefined,
                        endCallMessage: document.getElementById('endCallMessage').value || undefined
                    };
                    
                    // Update system prompt - preserve existing model config
                    const systemPrompt = document.getElementById('systemPrompt').value;
                    if (systemPrompt && currentAssistant.model) {
                        updates.model = {
                            ...currentAssistant.model,
                            messages: [{
                                role: 'system',
                                content: systemPrompt
                            }]
                        };
                    }
                    
                    // Update analysis plan if prompts provided
                    const successPrompt = document.getElementById('successEvalPrompt').value;
                    const summaryPrompt = document.getElementById('summaryPrompt').value;
                    
                    if (successPrompt || summaryPrompt) {
                        updates.analysisPlan = currentAssistant.analysisPlan || {};
                        
                        if (successPrompt) {
                            updates.analysisPlan.successEvaluationPlan = {
                                enabled: true,
                                messages: [{
                                    role: 'system',
                                    content: successPrompt
                                }]
                            };
                        }
                        
                        if (summaryPrompt) {
                            updates.analysisPlan.summaryPlan = {
                                enabled: true,
                                messages: [{
                                    role: 'system',
                                    content: summaryPrompt
                                }]
                            };
                        }
                    }
                    
                    const response = await fetch('vapi_api.php', {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/json' },
                        body: JSON.stringify({
                            action: 'update_assistant',
                            assistant_id: assistantId,
                            updates: updates
                        })
                    });
                    
                    const result = await response.json();
                    
                    if (result.error) {
                        showMessage('خطأ: ' + (result.message || JSON.stringify(result)), 'error');
                        return;
                    }
                    
                    showMessage('تم التحديث بنجاح! ✅', 'success');
                    closeModal();
                    loadAssistants();
                    
                } catch (error) {
                    showMessage('خطأ: ' + error.message, 'error');
                }
            } else {
                // CREATE MODE - Clone from maharah template
                try {
                    // First, get maharah assistant as template
                    const maharahResponse = await fetch('vapi_api.php', {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/json' },
                        body: JSON.stringify({ action: 'list_assistants', limit: 100 })
                    });
                    
                    const allAssistants = await maharahResponse.json();
                    const maharahTemplate = allAssistants.find(a => a.name && a.name.toLowerCase() === 'maharah');
                    
                    if (!maharahTemplate) {
                        showMessage('خطأ: لم يتم العثور على قالب maharah. يرجى التأكد من وجود مساعد باسم "maharah"', 'error');
                        return;
                    }
                    
                    // Clone maharah config
                    const newConfig = {
                        // User inputs from form
                        name: document.getElementById('name').value,
                        firstMessage: document.getElementById('firstMessage').value || maharahTemplate.firstMessage,
                        endCallMessage: document.getElementById('endCallMessage').value || maharahTemplate.endCallMessage,
                        
                        // Clone model config from maharah
                        model: {
                            ...maharahTemplate.model,
                            messages: [{
                                role: 'system',
                                content: document.getElementById('systemPrompt').value || maharahTemplate.model?.messages?.[0]?.content || 'You are a helpful assistant.'
                            }]
                        },
                        
                        // Clone voice from maharah
                        voice: maharahTemplate.voice,
                        
                        // Clone transcriber from maharah
                        transcriber: maharahTemplate.transcriber,
                        
                        // Clone other settings
                        recordingEnabled: maharahTemplate.recordingEnabled,
                        serverUrl: maharahTemplate.serverUrl,
                        voicemailMessage: maharahTemplate.voicemailMessage,
                        silenceTimeoutSeconds: maharahTemplate.silenceTimeoutSeconds,
                        maxDurationSeconds: maharahTemplate.maxDurationSeconds,
                        backgroundSound: maharahTemplate.backgroundSound,
                        backgroundDenoisingEnabled: maharahTemplate.backgroundDenoisingEnabled
                    };
                    
                    // Add analysis plan if provided
                    const successPrompt = document.getElementById('successEvalPrompt').value;
                    const summaryPrompt = document.getElementById('summaryPrompt').value;
                    
                    if (successPrompt || summaryPrompt || maharahTemplate.analysisPlan) {
                        newConfig.analysisPlan = maharahTemplate.analysisPlan || {};
                        
                        if (successPrompt) {
                            newConfig.analysisPlan.successEvaluationPlan = {
                                enabled: true,
                                messages: [{
                                    role: 'system',
                                    content: successPrompt
                                }]
                            };
                        } else if (maharahTemplate.analysisPlan?.successEvaluationPlan) {
                            newConfig.analysisPlan.successEvaluationPlan = maharahTemplate.analysisPlan.successEvaluationPlan;
                        }
                        
                        if (summaryPrompt) {
                            newConfig.analysisPlan.summaryPlan = {
                                enabled: true,
                                messages: [{
                                    role: 'system',
                                    content: summaryPrompt
                                }]
                            };
                        } else if (maharahTemplate.analysisPlan?.summaryPlan) {
                            newConfig.analysisPlan.summaryPlan = maharahTemplate.analysisPlan.summaryPlan;
                        }
                    }
                    
                    // Create new assistant
                    const response = await fetch('vapi_api.php', {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/json' },
                        body: JSON.stringify({
                            action: 'create_assistant',
                            config: newConfig
                        })
                    });
                    
                    const result = await response.json();
                    
                    if (result.error) {
                        showMessage('خطأ: ' + (result.message || JSON.stringify(result)), 'error');
                        return;
                    }
                    
                    showMessage('تم إنشاء المساعد بنجاح! ✅', 'success');
                    closeModal();
                    loadAssistants();
                    
                } catch (error) {
                    showMessage('خطأ: ' + error.message, 'error');
                }
            }
        }
        
        async function confirmDelete(assistantId, name) {
            if (!confirm('هل تريد حذف المساعد "' + name + '"؟\n\nهذا الإجراء لا يمكن التراجع عنه!')) {
                return;
            }
            
            try {
                const response = await fetch('vapi_api.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({
                        action: 'delete_assistant',
                        assistant_id: assistantId
                    })
                });
                
                const result = await response.json();
                
                if (result.error) {
                    showMessage('خطأ: ' + (result.message || JSON.stringify(result)), 'error');
                    return;
                }
                
                showMessage('تم الحذف بنجاح! 🗑️', 'success');
                loadAssistants();
                
            } catch (error) {
                showMessage('خطأ: ' + error.message, 'error');
            }
        }
        
        // Load assistants on page load
        document.addEventListener('DOMContentLoaded', function() {
            loadAssistants();
        });
        
        // Close modal when clicking outside
        window.onclick = function(event) {
            const modal = document.getElementById('assistantModal');
            if (event.target == modal) {
                closeModal();
            }
        }
    </script>
</body>
</html>