<?php
// transcript_editor.php - Edit transcripts stored in your database
require_once __DIR__ . '/vapi_config.php';

$pdo = new PDO(DB_DSN, DB_USER, DB_PASS, [
    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
    PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4"
]);

// Handle AJAX requests
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    header('Content-Type: application/json');
    
    $input = json_decode(file_get_contents('php://input'), true);
    $action = $input['action'] ?? '';
    
    try {
        switch($action) {
            case 'get_transcript':
                $call_id = $input['call_id'] ?? null;
                if (!$call_id) throw new Exception('Missing call_id');
                
                $stmt = $pdo->prepare("
                    SELECT id, call_id, role, message, timestamp, sequence_number 
                    FROM vapi_call_transcripts 
                    WHERE call_id = ? 
                    ORDER BY sequence_number
                ");
                $stmt->execute([$call_id]);
                echo json_encode($stmt->fetchAll(PDO::FETCH_ASSOC));
                break;
                
            case 'update_message':
                $id = $input['id'] ?? null;
                $new_message = $input['message'] ?? null;
                
                if (!$id || $new_message === null) {
                    throw new Exception('Missing id or message');
                }
                
                $stmt = $pdo->prepare("
                    UPDATE vapi_call_transcripts 
                    SET message = ?, updated_at = NOW() 
                    WHERE id = ?
                ");
                $stmt->execute([$new_message, $id]);
                
                echo json_encode(['success' => true]);
                break;
                
            case 'delete_message':
                $id = $input['id'] ?? null;
                if (!$id) throw new Exception('Missing id');
                
                $stmt = $pdo->prepare("DELETE FROM vapi_call_transcripts WHERE id = ?");
                $stmt->execute([$id]);
                
                echo json_encode(['success' => true]);
                break;
                
            case 'add_message':
                $call_id = $input['call_id'] ?? null;
                $role = $input['role'] ?? null;
                $message = $input['message'] ?? null;
                
                if (!$call_id || !$role || !$message) {
                    throw new Exception('Missing required fields');
                }
                
                // Get max sequence number
                $stmt = $pdo->prepare("
                    SELECT COALESCE(MAX(sequence_number), -1) + 1 as next_seq 
                    FROM vapi_call_transcripts 
                    WHERE call_id = ?
                ");
                $stmt->execute([$call_id]);
                $next_seq = $stmt->fetchColumn();
                
                // Insert new message
                $stmt = $pdo->prepare("
                    INSERT INTO vapi_call_transcripts 
                    (call_id, role, message, sequence_number, timestamp) 
                    VALUES (?, ?, ?, ?, NOW())
                ");
                $stmt->execute([$call_id, $role, $message, $next_seq]);
                
                echo json_encode([
                    'success' => true,
                    'id' => $pdo->lastInsertId()
                ]);
                break;
                
            default:
                throw new Exception('Unknown action');
        }
    } catch (Exception $e) {
        http_response_code(400);
        echo json_encode(['error' => $e->getMessage()]);
    }
    exit;
}

// Get call_id from URL
$call_id = $_GET['call_id'] ?? null;
if (!$call_id) {
    die('Missing call_id parameter');
}

// Get call info
$stmt = $pdo->prepare("SELECT * FROM vapi_call_logs WHERE call_id = ?");
$stmt->execute([$call_id]);
$call = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$call) {
    die('Call not found');
}
?>
<!DOCTYPE html>
<html dir="rtl" lang="ar">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>تحرير المحادثة - <?php echo htmlspecialchars($call_id); ?></title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: #f5f5f5;
            direction: rtl;
            padding: 20px;
        }
        .container {
            max-width: 900px;
            margin: 0 auto;
            background: white;
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        h1 {
            color: #333;
            margin-bottom: 10px;
        }
        .call-info {
            background: #f8f9fa;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
        }
        .call-info strong {
            color: #555;
        }
        .transcript {
            margin-top: 20px;
        }
        .message {
            margin-bottom: 15px;
            padding: 15px;
            border-radius: 8px;
            border: 1px solid #ddd;
            position: relative;
        }
        .message.assistant {
            background: #e3f2fd;
            border-right: 4px solid #2196f3;
        }
        .message.user {
            background: #f1f8e9;
            border-right: 4px solid #4caf50;
        }
        .message-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 10px;
        }
        .message-role {
            font-weight: bold;
            color: #333;
        }
        .message-actions {
            display: flex;
            gap: 5px;
        }
        .btn {
            padding: 5px 12px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-size: 12px;
        }
        .btn-edit {
            background: #007bff;
            color: white;
        }
        .btn-delete {
            background: #dc3545;
            color: white;
        }
        .btn-save {
            background: #28a745;
            color: white;
        }
        .btn-cancel {
            background: #6c757d;
            color: white;
        }
        .message-text {
            white-space: pre-wrap;
            line-height: 1.6;
        }
        .message-edit {
            width: 100%;
            min-height: 80px;
            padding: 10px;
            border: 1px solid #ddd;
            border-radius: 4px;
            font-family: inherit;
            font-size: 14px;
        }
        .add-message {
            margin-top: 20px;
            padding: 20px;
            background: #f8f9fa;
            border-radius: 8px;
        }
        .add-message h3 {
            margin-bottom: 15px;
        }
        .form-group {
            margin-bottom: 15px;
        }
        .form-group label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
        }
        .form-group select,
        .form-group textarea {
            width: 100%;
            padding: 10px;
            border: 1px solid #ddd;
            border-radius: 4px;
            font-family: inherit;
        }
        .alert {
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
        }
        .alert-success {
            background: #d4edda;
            color: #155724;
        }
        .alert-error {
            background: #f8d7da;
            color: #721c24;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>📝 تحرير محادثة المكالمة</h1>
        
        <div class="call-info">
            <strong>معرف المكالمة:</strong> <?php echo htmlspecialchars($call_id); ?><br>
            <strong>اسم المساعد:</strong> <?php echo htmlspecialchars($call['assistant_name'] ?? 'N/A'); ?><br>
            <strong>رقم العميل:</strong> <?php echo htmlspecialchars($call['customer_phone'] ?? 'N/A'); ?><br>
            <strong>وقت المكالمة:</strong> <?php echo htmlspecialchars($call['start_time'] ?? 'N/A'); ?>
        </div>

        <div id="alert-container"></div>

        <div class="transcript" id="transcript-container">
            <div style="text-align: center; padding: 40px; color: #666;">
                جاري تحميل المحادثة...
            </div>
        </div>

        <div class="add-message">
            <h3>➕ إضافة رسالة جديدة</h3>
            <form id="add-message-form" onsubmit="addMessage(event)">
                <div class="form-group">
                    <label>نوع الرسالة:</label>
                    <select id="new-role">
                        <option value="assistant">المساعد</option>
                        <option value="user">المستخدم</option>
                    </select>
                </div>
                <div class="form-group">
                    <label>نص الرسالة:</label>
                    <textarea id="new-message" rows="3" required></textarea>
                </div>
                <button type="submit" class="btn btn-save">إضافة</button>
            </form>
        </div>
    </div>

    <script>
        const callId = '<?php echo addslashes($call_id); ?>';
        let transcript = [];

        // Load transcript on page load
        async function loadTranscript() {
            try {
                const response = await fetch('transcript_editor.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({
                        action: 'get_transcript',
                        call_id: callId
                    })
                });

                transcript = await response.json();
                renderTranscript();
            } catch (error) {
                showAlert('خطأ في تحميل المحادثة: ' + error.message, 'error');
            }
        }

        // Render transcript
        function renderTranscript() {
            const container = document.getElementById('transcript-container');
            
            if (transcript.length === 0) {
                container.innerHTML = '<div style="text-align: center; padding: 40px; color: #666;">لا توجد رسائل في المحادثة</div>';
                return;
            }

            container.innerHTML = transcript.map(msg => `
                <div class="message ${msg.role}" id="message-${msg.id}">
                    <div class="message-header">
                        <span class="message-role">
                            ${msg.role === 'assistant' ? '🤖 المساعد' : '👤 المستخدم'}
                        </span>
                        <div class="message-actions">
                            <button class="btn btn-edit" onclick="editMessage(${msg.id})">✏️ تعديل</button>
                            <button class="btn btn-delete" onclick="deleteMessage(${msg.id})">🗑️ حذف</button>
                        </div>
                    </div>
                    <div class="message-content">
                        <div class="message-text" id="text-${msg.id}">${escapeHtml(msg.message)}</div>
                        <textarea class="message-edit" id="edit-${msg.id}" style="display: none;">${escapeHtml(msg.message)}</textarea>
                    </div>
                    <div class="message-actions" id="actions-${msg.id}" style="display: none; margin-top: 10px;">
                        <button class="btn btn-save" onclick="saveMessage(${msg.id})">💾 حفظ</button>
                        <button class="btn btn-cancel" onclick="cancelEdit(${msg.id})">إلغاء</button>
                    </div>
                </div>
            `).join('');
        }

        // Edit message
        function editMessage(id) {
            document.getElementById(`text-${id}`).style.display = 'none';
            document.getElementById(`edit-${id}`).style.display = 'block';
            document.getElementById(`actions-${id}`).style.display = 'block';
        }

        // Cancel edit
        function cancelEdit(id) {
            document.getElementById(`text-${id}`).style.display = 'block';
            document.getElementById(`edit-${id}`).style.display = 'none';
            document.getElementById(`actions-${id}`).style.display = 'none';
        }

        // Save edited message
        async function saveMessage(id) {
            const newMessage = document.getElementById(`edit-${id}`).value;
            
            try {
                const response = await fetch('transcript_editor.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({
                        action: 'update_message',
                        id: id,
                        message: newMessage
                    })
                });

                const result = await response.json();
                
                if (result.error) {
                    showAlert('خطأ: ' + result.error, 'error');
                    return;
                }

                showAlert('تم التحديث بنجاح', 'success');
                
                // Update in local array
                const msgIndex = transcript.findIndex(m => m.id === id);
                if (msgIndex !== -1) {
                    transcript[msgIndex].message = newMessage;
                }
                
                // Update display
                document.getElementById(`text-${id}`).textContent = newMessage;
                cancelEdit(id);
                
            } catch (error) {
                showAlert('خطأ: ' + error.message, 'error');
            }
        }

        // Delete message
        async function deleteMessage(id) {
            if (!confirm('هل تريد حذف هذه الرسالة؟')) return;
            
            try {
                const response = await fetch('transcript_editor.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({
                        action: 'delete_message',
                        id: id
                    })
                });

                const result = await response.json();
                
                if (result.error) {
                    showAlert('خطأ: ' + result.error, 'error');
                    return;
                }

                showAlert('تم الحذف بنجاح', 'success');
                
                // Remove from local array
                transcript = transcript.filter(m => m.id !== id);
                renderTranscript();
                
            } catch (error) {
                showAlert('خطأ: ' + error.message, 'error');
            }
        }

        // Add new message
        async function addMessage(event) {
            event.preventDefault();
            
            const role = document.getElementById('new-role').value;
            const message = document.getElementById('new-message').value;
            
            try {
                const response = await fetch('transcript_editor.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({
                        action: 'add_message',
                        call_id: callId,
                        role: role,
                        message: message
                    })
                });

                const result = await response.json();
                
                if (result.error) {
                    showAlert('خطأ: ' + result.error, 'error');
                    return;
                }

                showAlert('تم الإضافة بنجاح', 'success');
                
                // Clear form
                document.getElementById('new-message').value = '';
                
                // Reload transcript
                await loadTranscript();
                
            } catch (error) {
                showAlert('خطأ: ' + error.message, 'error');
            }
        }

        // Show alert message
        function showAlert(message, type = 'success') {
            const container = document.getElementById('alert-container');
            container.innerHTML = `<div class="alert alert-${type}">${message}</div>`;
            setTimeout(() => container.innerHTML = '', 5000);
        }

        // Escape HTML
        function escapeHtml(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }

        // Load on page load
        window.addEventListener('DOMContentLoaded', loadTranscript);
    </script>
</body>
</html>