<?php
// Reliable XLSX export for analysis data
require_once __DIR__ . '/vapi_config.php';

session_start();
if (!isset($_SESSION['logged_in'])) {
    header('Location: call_logs.php');
    exit;
}

function h($s) { return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

// Check server capabilities
$has_zip = class_exists('ZipArchive');
$temp_writable = is_writable(sys_get_temp_dir());

// Database connection
try {
    $pdo = new PDO(DB_DSN, DB_USER, DB_PASS, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4"
    ]);
} catch (Exception $e) {
    die("Database connection failed");
}

// Simple filters
$search = trim($_GET['search'] ?? '');
$from_date = trim($_GET['from_date'] ?? '');
$to_date = trim($_GET['to_date'] ?? '');

$where_parts = [];
$params = [];

if ($search) {
    $where_parts[] = "(ca.call_id LIKE ? OR cl.customer_phone LIKE ?)";
    $params[] = '%' . $search . '%';
    $params[] = '%' . $search . '%';
}

if ($from_date && preg_match('/^\d{4}-\d{2}-\d{2}$/', $from_date)) {
    $where_parts[] = "ca.analyzed_at >= ?";
    $params[] = $from_date . " 00:00:00";
}

if ($to_date && preg_match('/^\d{4}-\d{2}-\d{2}$/', $to_date)) {
    $where_parts[] = "ca.analyzed_at <= ?";
    $params[] = $to_date . " 23:59:59";
}

$where_sql = !empty($where_parts) ? 'WHERE ' . implode(' AND ', $where_parts) : '';

// Function to create XLSX file
function createXLSXFile($data, $filters = []) {
    if (!class_exists('ZipArchive')) {
        throw new Exception('ZipArchive not available');
    }
    
    $filename = 'call_analysis_' . date('Y-m-d_H-i-s') . '.xlsx';
    $temp_dir = sys_get_temp_dir() . '/xlsx_' . uniqid();
    
    // Create directories
    if (!mkdir($temp_dir, 0755, true)) {
        throw new Exception('Cannot create temp directory');
    }
    
    $dirs = ['xl', 'xl/worksheets', '_rels', 'xl/_rels'];
    foreach ($dirs as $dir) {
        if (!mkdir($temp_dir . '/' . $dir, 0755, true)) {
            throw new Exception('Cannot create directory: ' . $dir);
        }
    }
    
    // Create required XML files
    file_put_contents($temp_dir . '/[Content_Types].xml', 
        '<?xml version="1.0" encoding="UTF-8" standalone="yes"?>' .
        '<Types xmlns="http://schemas.openxmlformats.org/package/2006/content-types">' .
        '<Default Extension="rels" ContentType="application/vnd.openxmlformats-package.relationships+xml"/>' .
        '<Default Extension="xml" ContentType="application/xml"/>' .
        '<Override PartName="/xl/workbook.xml" ContentType="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet.main+xml"/>' .
        '<Override PartName="/xl/worksheets/sheet1.xml" ContentType="application/vnd.openxmlformats-officedocument.spreadsheetml.worksheet+xml"/>' .
        '</Types>'
    );
    
    file_put_contents($temp_dir . '/_rels/.rels',
        '<?xml version="1.0" encoding="UTF-8" standalone="yes"?>' .
        '<Relationships xmlns="http://schemas.openxmlformats.org/package/2006/relationships">' .
        '<Relationship Id="rId1" Type="http://schemas.openxmlformats.org/officeDocument/2006/relationships/officeDocument" Target="xl/workbook.xml"/>' .
        '</Relationships>'
    );
    
    file_put_contents($temp_dir . '/xl/_rels/workbook.xml.rels',
        '<?xml version="1.0" encoding="UTF-8" standalone="yes"?>' .
        '<Relationships xmlns="http://schemas.openxmlformats.org/package/2006/relationships">' .
        '<Relationship Id="rId1" Type="http://schemas.openxmlformats.org/officeDocument/2006/relationships/worksheet" Target="worksheets/sheet1.xml"/>' .
        '</Relationships>'
    );
    
    file_put_contents($temp_dir . '/xl/workbook.xml',
        '<?xml version="1.0" encoding="UTF-8" standalone="yes"?>' .
        '<workbook xmlns="http://schemas.openxmlformats.org/spreadsheetml/2006/main" xmlns:r="http://schemas.openxmlformats.org/officeDocument/2006/relationships">' .
        '<sheets><sheet name="Call Analysis" sheetId="1" r:id="rId1"/></sheets>' .
        '</workbook>'
    );
    
    // Create worksheet
    $xml = '<?xml version="1.0" encoding="UTF-8" standalone="yes"?>';
    $xml .= '<worksheet xmlns="http://schemas.openxmlformats.org/spreadsheetml/2006/main">';
    $xml .= '<sheetData>';
    
    $row = 1;
    
    // Add filter info if present
    if (!empty($filters)) {
        $xml .= '<row r="' . $row . '">';
        $xml .= '<c r="A' . $row . '" t="inlineStr"><is><t>Filters Applied: ' . implode(' | ', $filters) . '</t></is></c>';
        $xml .= '</row>';
        $row++;
        $xml .= '<row r="' . $row . '"></row>'; // Empty row
        $row++;
    }
    
    // Headers
    $headers = [
        'Call ID', 'Assistant Name', 'Customer Phone', 'Assistant Phone', 'Call Type',
        'Start Time', 'Duration (sec)', 'Cost USD', 'Success Eval', 'Customer Satisfaction',
        'Call Result', 'Next Action', 'Follow Up Date', 'Should Call Again', 'Customer Needs',
        'Agent Performance', 'Key Points', 'Analysis Summary', 'Main Status', 'Sub Status', 'Analyzed At'
    ];
    
    $cols = range('A', 'U');
    
    $xml .= '<row r="' . $row . '">';
    for ($i = 0; $i < count($headers); $i++) {
        $xml .= '<c r="' . $cols[$i] . $row . '" t="inlineStr"><is><t>' . htmlspecialchars($headers[$i], ENT_XML1) . '</t></is></c>';
    }
    $xml .= '</row>';
    $row++;
    
    // Data rows
    foreach ($data as $record) {
        $values = [
            $record['call_id'] ?? '',
            $record['assistant_name'] ?? '',
            $record['customer_phone'] ?? '',
            $record['assistant_phone'] ?? '',
            $record['call_type'] ?? '',
            $record['start_time'] ?? '',
            $record['duration_seconds'] ?? '',
            $record['cost_usd'] ?? '',
            $record['success_eval'] ?? '',
            $record['customer_satisfaction'] ?? '',
            $record['call_result'] ?? '',
            $record['next_action'] ?? '',
            $record['follow_up_date'] ?? '',
            $record['should_call_again'] ? 'Yes' : 'No',
            $record['customer_needs'] ?? '',
            $record['agent_performance'] ?? '',
            $record['key_points'] ?? '',
            $record['analysis_summary'] ?? '',
            $record['main_status'] ?? '',
            $record['sub_status'] ?? '',
            $record['analyzed_at'] ?? ''
        ];
        
        $xml .= '<row r="' . $row . '">';
        for ($i = 0; $i < count($values); $i++) {
            $clean_value = htmlspecialchars(strip_tags($values[$i]), ENT_XML1);
            $xml .= '<c r="' . $cols[$i] . $row . '" t="inlineStr"><is><t>' . $clean_value . '</t></is></c>';
        }
        $xml .= '</row>';
        $row++;
    }
    
    $xml .= '</sheetData>';
    $xml .= '</worksheet>';
    
    file_put_contents($temp_dir . '/xl/worksheets/sheet1.xml', $xml);
    
    // Create ZIP
    $zip = new ZipArchive();
    $zip_file = $temp_dir . '/output.xlsx';
    
    if ($zip->open($zip_file, ZipArchive::CREATE) !== TRUE) {
        throw new Exception('Cannot create ZIP file');
    }
    
    // Add files to ZIP
    $iterator = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($temp_dir, FilesystemIterator::SKIP_DOTS),
        RecursiveIteratorIterator::LEAVES_ONLY
    );
    
    foreach ($iterator as $file) {
        if ($file->getFilename() !== 'output.xlsx') {
            $filePath = $file->getRealPath();
            $relativePath = substr($filePath, strlen($temp_dir) + 1);
            $zip->addFile($filePath, $relativePath);
        }
    }
    
    $zip->close();
    
    // Send file
    header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    header('Content-Length: ' . filesize($zip_file));
    
    readfile($zip_file);
    
    // Cleanup
    $iterator = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($temp_dir, FilesystemIterator::SKIP_DOTS),
        RecursiveIteratorIterator::CHILD_FIRST
    );
    
    foreach ($iterator as $file) {
        if ($file->isDir()) {
            rmdir($file->getRealPath());
        } else {
            unlink($file->getRealPath());
        }
    }
    rmdir($temp_dir);
}

// Handle export
if (isset($_GET['action']) && $_GET['action'] === 'export') {
    try {
        $sql = "
            SELECT 
                ca.call_id,
                cl.assistant_name,
                cl.customer_phone,
                cl.assistant_phone,
                cl.call_type,
                cl.start_time,
                cl.duration_seconds,
                cl.cost_usd,
                cl.success_eval,
                ca.customer_satisfaction,
                ca.call_result,
                ca.next_action,
                ca.follow_up_date,
                ca.should_call_again,
                ca.customer_needs,
                ca.agent_performance,
                ca.key_points,
                ca.analysis_summary,
                ca.main_status,
                ca.sub_status,
                ca.analyzed_at
            FROM vapi_call_analysiss ca
            LEFT JOIN vapi_call_logs cl ON ca.call_id = cl.call_id
            $where_sql
            ORDER BY ca.analyzed_at DESC
            LIMIT 3000
        ";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $data = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $filters = [];
        if ($search) $filters[] = "Search: $search";
        if ($from_date) $filters[] = "From: $from_date";
        if ($to_date) $filters[] = "To: $to_date";
        
        createXLSXFile($data, $filters);
        exit;
        
    } catch (Exception $e) {
        $error = "Export failed: " . $e->getMessage();
    }
}

// Get count
try {
    $count_sql = "SELECT COUNT(*) FROM vapi_call_analysiss ca LEFT JOIN vapi_call_logs cl ON ca.call_id = cl.call_id $where_sql";
    $count_stmt = $pdo->prepare($count_sql);
    $count_stmt->execute($params);
    $total = $count_stmt->fetchColumn();
} catch (Exception $e) {
    $total = 0;
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="utf-8">
    <title>تصدير XLSX - تحليل المكالمات</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            margin: 20px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            direction: rtl;
        }
        .container {
            max-width: 1000px;
            margin: 0 auto;
            background: white;
            padding: 30px;
            border-radius: 15px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.2);
        }
        h1 {
            color: #2d3748;
            text-align: center;
            margin-bottom: 30px;
            font-size: 2rem;
        }
        .status-check {
            background: #f0f9ff;
            border: 2px solid #0ea5e9;
            border-radius: 8px;
            padding: 20px;
            margin-bottom: 30px;
        }
        .status-check h3 {
            color: #0369a1;
            margin-bottom: 15px;
        }
        .status-item {
            display: flex;
            justify-content: space-between;
            margin: 10px 0;
            padding: 8px;
            background: white;
            border-radius: 5px;
        }
        .status-ok { border-right: 4px solid #22c55e; }
        .status-fail { border-right: 4px solid #ef4444; }
        
        .form-row {
            display: grid;
            grid-template-columns: 1fr 1fr 1fr auto;
            gap: 20px;
            margin-bottom: 20px;
            align-items: end;
        }
        .form-group label {
            display: block;
            margin-bottom: 8px;
            font-weight: bold;
            color: #4a5568;
        }
        .form-group input {
            width: 100%;
            padding: 12px;
            border: 2px solid #e2e8f0;
            border-radius: 8px;
            font-size: 14px;
        }
        .form-group input:focus {
            border-color: #3182ce;
            outline: none;
        }
        .btn {
            background: linear-gradient(135deg, #48bb78 0%, #38a169 100%);
            color: white;
            padding: 14px 28px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-size: 16px;
            font-weight: bold;
            text-decoration: none;
            display: inline-block;
            transition: transform 0.2s;
        }
        .btn:hover { transform: translateY(-2px); }
        .btn-primary { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); }
        .btn-clear { background: linear-gradient(135deg, #f56565 0%, #e53e3e 100%); }
        
        .stats {
            background: linear-gradient(135deg, #ecfdf5 0%, #d1fae5 100%);
            border: 2px solid #34d399;
            border-radius: 10px;
            padding: 25px;
            text-align: center;
            margin: 25px 0;
        }
        .stats h3 {
            color: #047857;
            margin-bottom: 10px;
        }
        .actions {
            text-align: center;
            margin: 30px 0;
        }
        .error {
            background: #fef2f2;
            color: #dc2626;
            padding: 15px;
            border-radius: 8px;
            margin: 20px 0;
            border: 2px solid #f87171;
        }
        .nav {
            text-align: center;
            margin-top: 40px;
            padding-top: 20px;
            border-top: 2px solid #e5e7eb;
        }
        @media (max-width: 768px) {
            .form-row { grid-template-columns: 1fr; }
            .container { margin: 10px; padding: 20px; }
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>تصدير XLSX - تحليل المكالمات</h1>
        
        <!-- Server Status Check -->
        <div class="status-check">
            <h3>حالة الخادم</h3>
            <div class="status-item <?= $has_zip ? 'status-ok' : 'status-fail' ?>">
                <span>ZipArchive Extension</span>
                <span><?= $has_zip ? '✅ متوفر' : '❌ غير متوفر' ?></span>
            </div>
            <div class="status-item <?= $temp_writable ? 'status-ok' : 'status-fail' ?>">
                <span>Temporary Directory Writable</span>
                <span><?= $temp_writable ? '✅ قابل للكتابة' : '❌ غير قابل للكتابة' ?></span>
            </div>
            <div class="status-item status-ok">
                <span>PHP Version</span>
                <span><?= PHP_VERSION ?></span>
            </div>
        </div>
        
        <?php if (isset($error)): ?>
        <div class="error">خطأ: <?= h($error) ?></div>
        <?php endif; ?>
        
        <?php if (!$has_zip): ?>
        <div class="error">
            <strong>تحذير:</strong> ZipArchive غير متوفر على الخادم. لن يتمكن من إنشاء ملفات XLSX.
            اتصل بمزود الاستضافة لتفعيل PHP ZIP extension.
        </div>
        <?php endif; ?>
        
        <!-- Filters -->
        <form method="get">
            <div class="form-row">
                <div class="form-group">
                    <label>البحث</label>
                    <input type="text" name="search" placeholder="معرف المكالمة أو رقم العميل" value="<?= h($search) ?>">
                </div>
                
                <div class="form-group">
                    <label>من تاريخ</label>
                    <input type="date" name="from_date" value="<?= h($from_date) ?>">
                </div>
                
                <div class="form-group">
                    <label>إلى تاريخ</label>
                    <input type="date" name="to_date" value="<?= h($to_date) ?>">
                </div>
                
                <div class="form-group">
                    <button type="submit" class="btn btn-primary">تطبيق المرشحات</button>
                </div>
            </div>
        </form>
        
        <!-- Stats -->
        <div class="stats">
            <h3>عدد السجلات: <?= number_format($total) ?></h3>
            <p>سيتم تصدير أول 3000 سجل للحفاظ على الأداء</p>
        </div>
        
        <!-- Actions -->
        <div class="actions">
            <?php if ($total > 0 && $has_zip): ?>
            <a href="?<?= http_build_query(array_merge($_GET, ['action' => 'export'])) ?>" class="btn">
                تصدير إلى XLSX (<?= number_format(min($total, 3000)) ?> سجل)
            </a>
            <?php elseif (!$has_zip): ?>
            <p style="color: #dc2626; font-weight: bold;">
                لا يمكن التصدير - ZipArchive غير متوفر
            </p>
            <?php else: ?>
            <p style="color: #6b7280;">لا توجد بيانات للتصدير</p>
            <?php endif; ?>
            
            <a href="?" class="btn btn-clear">مسح المرشحات</a>
        </div>
        
        <!-- Navigation -->
        <div class="nav">
            <a href="call_logs.php" class="btn btn-primary">سجل المكالمات</a>
            <a href="call_analyzer.php" class="btn btn-primary">محلل المكالمات</a>
            <a href="analysis_dashboard.php" class="btn btn-primary">لوحة التحليل</a>
        </div>
    </div>

    <script>
        // Show loading state on export
        document.addEventListener('DOMContentLoaded', function() {
            const exportLink = document.querySelector('a[href*="action=export"]');
            if (exportLink) {
                exportLink.addEventListener('click', function(e) {
                    this.innerHTML = 'جاري إنشاء ملف XLSX...';
                    this.style.pointerEvents = 'none';
                    
                    setTimeout(() => {
                        this.innerHTML = 'تصدير إلى XLSX';
                        this.style.pointerEvents = 'auto';
                    }, 10000);
                });
            }
        });
    </script>
</body>
</html>