<?php
// analysis_dashboard.php - Fixed version
require_once __DIR__ . '/vapi_config.php';

session_start();

// Simple authentication (use same as call logs)
$users = [
    'admin' => password_hash('CallLog2024!', PASSWORD_DEFAULT),
    'manager' => password_hash('Manager@456', PASSWORD_DEFAULT),
    'viewer' => password_hash('Viewer#789', PASSWORD_DEFAULT)
];

if (!isset($_SESSION['logged_in'])) {
    header('Location: call_logs.php');
    exit;
}

try {
    $pdo = new PDO(DB_DSN, DB_USER, DB_PASS, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4"
    ]);
} catch (Exception $e) {
    die("Database connection failed: " . $e->getMessage());
}

// Handle manual analysis trigger
if (isset($_POST['analyze_call_id'])) {
    try {
        // Simple redirect to call_analyzer.php for now
        $call_id = $_POST['analyze_call_id'];
        header("Location: call_analyzer.php?action=analyze_single&call_id=" . urlencode($call_id));
        exit;
    } catch (Exception $e) {
        $error_message = "فشل في تحليل المكالمة: " . $e->getMessage();
    }
}

// Get analysis statistics
try {
    $stats_query = "SELECT 
                        COUNT(*) as total_analyzed,
                        COUNT(CASE WHEN customer_satisfaction IN ('راضي', 'راضي جداً') THEN 1 END) as satisfied_customers,
                        COUNT(CASE WHEN call_result = 'نجح' THEN 1 END) as successful_calls,
                        COUNT(CASE WHEN should_call_again = 1 THEN 1 END) as needs_followup,
                        COUNT(CASE WHEN agent_performance IN ('ممتاز', 'جيد') THEN 1 END) as good_performance,
                        COUNT(CASE WHEN main_status IS NOT NULL THEN 1 END) as has_company_status,
                        COUNT(CASE WHEN main_status = 'تم السداد' THEN 1 END) as payment_completed,
                        COUNT(CASE WHEN main_status = 'العميل وعد بالسداد' THEN 1 END) as payment_promised
                    FROM vapi_call_analysiss";
    
    $stats = $pdo->query($stats_query)->fetch(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    $stats = [
        'total_analyzed' => 0,
        'satisfied_customers' => 0,
        'successful_calls' => 0,
        'needs_followup' => 0,
        'good_performance' => 0,
        'has_company_status' => 0,
        'payment_completed' => 0,
        'payment_promised' => 0
    ];
}

// Get recent analyses
try {
    $recent_analyses = $pdo->query("
        SELECT ca.*, cl.assistant_name, cl.customer_phone, cl.start_time, cl.duration_seconds
        FROM vapi_call_analysiss ca
        JOIN vapi_call_logs cl ON ca.call_id = cl.call_id
        ORDER BY ca.analyzed_at DESC
        LIMIT 20
    ")->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    $recent_analyses = [];
}

// Get status statistics
try {
    $status_stats = $pdo->query("
        SELECT 
            main_status,
            COUNT(*) as count
        FROM vapi_call_analysiss 
        WHERE main_status IS NOT NULL
        GROUP BY main_status
        ORDER BY count DESC
        LIMIT 10
    ")->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    $status_stats = [];
}

// Get calls needing analysis
try {
    $unanalyzed_calls = $pdo->query("
        SELECT cl.call_id, cl.assistant_name, cl.customer_phone, cl.start_time, cl.duration_seconds
        FROM vapi_call_logs cl
        LEFT JOIN vapi_call_analysiss ca ON cl.call_id = ca.call_id
        WHERE ca.call_id IS NULL 
        AND cl.duration_seconds > 30
        AND EXISTS (SELECT 1 FROM vapi_call_transcripts WHERE call_id = cl.call_id)
        ORDER BY cl.start_time DESC
        LIMIT 10
    ")->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    $unanalyzed_calls = [];
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="utf-8">
    <title>لوحة تحليل المكالمات</title>
    <meta name="viewport" content="width=device-width,initial-scale=1">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        body { 
            font-family: 'Segoe UI', 'Cairo', Tahoma, Arial, sans-serif; 
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            direction: rtl;
            color: #333;
        }
        
        .header {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            padding: 20px 0;
            box-shadow: 0 2px 20px rgba(0,0,0,0.1);
            margin-bottom: 30px;
        }
        
        .container { max-width: 1400px; margin: 0 auto; padding: 0 20px; }
        
        h1 {
            text-align: center;
            color: #2d3748;
            font-size: 2.2rem;
            font-weight: 700;
            margin-bottom: 10px;
        }
        
        .subtitle {
            text-align: center;
            color: #718096;
            font-size: 1.1rem;
        }
        
        .card { 
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            border-radius: 20px; 
            padding: 30px; 
            margin: 25px 0; 
            box-shadow: 0 8px 32px rgba(0,0,0,0.1);
        }
        
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin: 25px 0;
        }
        
        .stat-card {
            background: linear-gradient(135deg, #f7fafc 0%, #edf2f7 100%);
            padding: 25px;
            border-radius: 16px;
            text-align: center;
            border-right: 4px solid #667eea;
        }
        
        .stat-number {
            font-size: 2.5rem;
            font-weight: 700;
            color: #2d3748;
            margin-bottom: 8px;
        }
        
        .stat-label {
            color: #718096;
            font-weight: 500;
        }
        
        .btn {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 12px 24px;
            border: none;
            border-radius: 10px;
            cursor: pointer;
            font-size: 14px;
            font-weight: 600;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 8px;
            transition: all 0.3s ease;
        }
        
        .btn:hover { transform: translateY(-2px); }
        
        .btn-small {
            padding: 8px 16px;
            font-size: 12px;
        }
        
        .btn-success {
            background: linear-gradient(135deg, #48bb78 0%, #38a169 100%);
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
            background: white;
            border-radius: 12px;
            overflow: hidden;
        }
        
        th, td {
            padding: 15px;
            text-align: right;
            border-bottom: 1px solid #e2e8f0;
        }
        
        th {
            background: #f8f9fa;
            font-weight: 700;
            color: #2d3748;
        }
        
        .status-badge {
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            text-align: center;
        }
        
        .status-satisfied { background: #c6f6d5; color: #22543d; }
        .status-neutral { background: #fed7d7; color: #742a2a; }
        .status-success { background: #c6f6d5; color: #22543d; }
        .status-failed { background: #fed7d7; color: #742a2a; }
        
        .alert {
            padding: 16px;
            border-radius: 8px;
            margin: 20px 0;
        }
        
        .alert-success {
            background: #c6f6d5;
            color: #22543d;
            border: 1px solid #9ae6b4;
        }
        
        .alert-error {
            background: #fed7d7;
            color: #742a2a;
            border: 1px solid #feb2b2;
        }
        
        .empty-state {
            text-align: center;
            padding: 40px;
            color: #718096;
        }
        
        @media (max-width: 768px) {
            .stats-grid { grid-template-columns: 1fr; }
            .container { padding: 0 15px; }
            h1 { font-size: 1.8rem; }
        }
    </style>
</head>
<body>
    <div class="header">
        <div class="container">
            <h1><i class="fas fa-chart-line"></i> لوحة تحليل المكالمات</h1>
            <p class="subtitle">تحليل ذكي لمكالمات خدمة العملاء</p>
        </div>
    </div>

    <div class="container">
        <?php if (isset($success_message)): ?>
        <div class="alert alert-success">
            <i class="fas fa-check-circle"></i> <?= $success_message ?>
        </div>
        <?php endif; ?>
        
        <?php if (isset($error_message)): ?>
        <div class="alert alert-error">
            <i class="fas fa-exclamation-triangle"></i> <?= $error_message ?>
        </div>
        <?php endif; ?>

        <!-- Statistics -->
        <div class="card">
            <h2><i class="fas fa-chart-bar"></i> إحصائيات التحليل</h2>
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-number"><?= number_format($stats['total_analyzed']) ?></div>
                    <div class="stat-label">مكالمة تم تحليلها</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number"><?= number_format($stats['satisfied_customers']) ?></div>
                    <div class="stat-label">عميل راضي</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number"><?= number_format($stats['successful_calls']) ?></div>
                    <div class="stat-label">مكالمة ناجحة</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number"><?= number_format($stats['needs_followup']) ?></div>
                    <div class="stat-label">يحتاج متابعة</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number"><?= number_format($stats['good_performance']) ?></div>
                    <div class="stat-label">أداء ممتاز/جيد</div>
                </div>
            </div>
        </div>

        <!-- Unanalyzed Calls -->
        <?php if (!empty($unanalyzed_calls)): ?>
        <div class="card">
            <h2><i class="fas fa-clock"></i> مكالمات في انتظار التحليل (<?= count($unanalyzed_calls) ?>)</h2>
            <div style="overflow-x: auto;">
                <table>
                    <thead>
                        <tr>
                            <th>معرف المكالمة</th>
                            <th>الوكيل</th>
                            <th>رقم العميل</th>
                            <th>وقت البداية</th>
                            <th>المدة</th>
                            <th>إجراء</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($unanalyzed_calls as $call): ?>
                        <tr>
                            <td><?= htmlspecialchars(substr($call['call_id'], 0, 8)) ?>...</td>
                            <td><?= htmlspecialchars($call['assistant_name']) ?></td>
                            <td><?= htmlspecialchars($call['customer_phone']) ?></td>
                            <td><?= $call['start_time'] ? date('M j, H:i', strtotime($call['start_time'])) : 'غير محدد' ?></td>
                            <td><?= $call['duration_seconds'] ? intval($call['duration_seconds']/60) . 'دقيقة' : 'غير محدد' ?></td>
                            <td>
                                <a href="call_analyzer.php?action=analyze_single&call_id=<?= urlencode($call['call_id']) ?>" class="btn btn-small btn-success">
                                    <i class="fas fa-play"></i> تحليل الآن
                                </a>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
        <?php endif; ?>

        <!-- Recent Analyses -->
        <div class="card">
            <h2><i class="fas fa-history"></i> التحليلات الأخيرة</h2>
            <?php if (!empty($recent_analyses)): ?>
            <div style="overflow-x: auto;">
                <table>
                    <thead>
                        <tr>
                            <th>معرف المكالمة</th>
                            <th>الوكيل</th>
                            <th>رقم العميل</th>
                            <th>رضا العميل</th>
                            <th>نتيجة المكالمة</th>
                            <th>أداء الوكيل</th>
                            <th>يحتاج متابعة</th>
                            <th>تاريخ التحليل</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($recent_analyses as $analysis): ?>
                        <tr>
                            <td><?= htmlspecialchars(substr($analysis['call_id'], 0, 8)) ?>...</td>
                            <td><?= htmlspecialchars($analysis['assistant_name']) ?></td>
                            <td><?= htmlspecialchars($analysis['customer_phone']) ?></td>
                            <td>
                                <span class="status-badge status-<?= strpos($analysis['customer_satisfaction'], 'راضي') !== false ? 'satisfied' : 'neutral' ?>">
                                    <?= htmlspecialchars($analysis['customer_satisfaction']) ?>
                                </span>
                            </td>
                            <td>
                                <span class="status-badge status-<?= $analysis['call_result'] == 'نجح' ? 'success' : 'failed' ?>">
                                    <?= htmlspecialchars($analysis['call_result']) ?>
                                </span>
                            </td>
                            <td><?= htmlspecialchars($analysis['agent_performance']) ?></td>
                            <td><?= $analysis['should_call_again'] ? '✅ نعم' : '❌ لا' ?></td>
                            <td><?= date('M j, H:i', strtotime($analysis['analyzed_at'])) ?></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            <?php else: ?>
            <div class="empty-state">
                <i class="fas fa-chart-line" style="font-size: 3rem; margin-bottom: 15px;"></i>
                <h3>لا توجد تحليلات حتى الآن</h3>
                <p>ستظهر نتائج التحليل هنا عند إجراء التحليل الأول</p>
            </div>
            <?php endif; ?>
        </div>

        <!-- Quick Actions -->
        <div class="card">
            <h2><i class="fas fa-tools"></i> إجراءات سريعة</h2>
            <div style="display: flex; gap: 15px; flex-wrap: wrap;">
                <a href="call_logs.php" class="btn">
                    <i class="fas fa-list"></i> عرض سجل المكالمات
                </a>
                <a href="call_analyzer.php" class="btn btn-success">
                    <i class="fas fa-play"></i> تشغيل التحليل التلقائي
                </a>
                <button onclick="location.reload()" class="btn">
                    <i class="fas fa-sync-alt"></i> تحديث البيانات
                </button>
            </div>
        </div>
    </div>

    <script>
        // Auto-refresh every 5 minutes
        setTimeout(function() {
            location.reload();
        }, 300000);
    </script>
</body>
</html>