<?php

namespace App\Services;

use App\Models\CrmAgentAssignment;
use App\Models\CrmSetting;
use App\Models\CrmUser;
use App\Models\CrmAuditLog;

class RoutingService
{
    public function getAssignedAgent($customerE164)
    {
        // Check if customer is already assigned to an agent
        $assignment = CrmAgentAssignment::where('customer_e164', $customerE164)->first();
        
        if ($assignment) {
            $agent = $assignment->agent;
            if ($agent && $agent->active) {
                return $agent;
            }
        }

        // If no assignment or agent is inactive, assign to default agent
        return $this->getDefaultAgent();
    }

    public function getDefaultAgent()
    {
        $settings = CrmSetting::getSettings();
        
        if ($settings->default_agent_id) {
            $agent = CrmUser::where('id', $settings->default_agent_id)
                           ->where('role', 'Agent')
                           ->where('active', true)
                           ->first();
            
            if ($agent) {
                return $agent;
            }
        }

        // If no default agent set, get the first available agent
        return CrmUser::where('role', 'Agent')
                     ->where('active', true)
                     ->first();
    }

    public function assignCustomerToAgent($customerE164, $agentId = null)
    {
        if (!$agentId) {
            $agent = $this->getDefaultAgent();
            $agentId = $agent ? $agent->id : null;
        }

        if (!$agentId) {
            throw new \Exception('No available agent to assign customer');
        }

        $assignment = CrmAgentAssignment::assignCustomerToAgent($customerE164, $agentId);
        
        CrmAuditLog::log('customer_assigned', [
            'customer_e164' => $customerE164,
            'agent_id' => $agentId
        ]);

        return $assignment;
    }

    public function reassignCustomer($customerE164, $newAgentId)
    {
        $oldAssignment = CrmAgentAssignment::where('customer_e164', $customerE164)->first();
        $oldAgentId = $oldAssignment ? $oldAssignment->agent_id : null;

        $assignment = CrmAgentAssignment::assignCustomerToAgent($customerE164, $newAgentId);
        
        CrmAuditLog::log('customer_reassigned', [
            'customer_e164' => $customerE164,
            'old_agent_id' => $oldAgentId,
            'new_agent_id' => $newAgentId
        ]);

        return $assignment;
    }

    public function getAgentWorkload($agentId)
    {
        return CrmAgentAssignment::where('agent_id', $agentId)->count();
    }

    public function getLeastBusyAgent()
    {
        return CrmUser::where('role', 'Agent')
                     ->where('active', true)
                     ->withCount('assignments')
                     ->orderBy('assignments_count', 'asc')
                     ->first();
    }

    public function autoAssignCustomer($customerE164)
    {
        // Try to get least busy agent for better load distribution
        $agent = $this->getLeastBusyAgent();
        
        if ($agent) {
            return $this->assignCustomerToAgent($customerE164, $agent->id);
        }

        // Fallback to default agent
        return $this->assignCustomerToAgent($customerE164);
    }
}
