<?php

namespace App\Services;

use App\Models\CrmSetting;
use App\Models\CrmTemplate;
use App\Models\CrmAuditLog;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class MetaService
{
    protected $baseUrl = 'https://graph.facebook.com/v18.0';
    protected $settings;

    public function __construct()
    {
        $this->settings = CrmSetting::getSettings();
    }

    public function getAccessToken()
    {
        return $this->settings->meta_access_token;
    }

    public function getPhoneNumberId()
    {
        return $this->settings->phone_number_id;
    }

    public function syncTemplates()
    {
        try {
            $phoneNumberId = $this->getPhoneNumberId();
            $accessToken = $this->getAccessToken();

            if (!$phoneNumberId || !$accessToken) {
                throw new \Exception('Phone Number ID or Access Token not configured');
            }

            $response = Http::get("{$this->baseUrl}/{$phoneNumberId}/message_templates", [
                'access_token' => $accessToken
            ]);

            if (!$response->successful()) {
                throw new \Exception('Failed to fetch templates: ' . $response->body());
            }

            $data = $response->json();
            $templates = $data['data'] ?? [];

            foreach ($templates as $template) {
                CrmTemplate::updateOrCreate(
                    [
                        'name' => $template['name'],
                        'language' => $template['language']
                    ],
                    [
                        'category' => $template['category'] ?? 'unknown',
                        'template_data' => $template,
                        'synced_at' => now()
                    ]
                );
            }

            CrmAuditLog::log('templates_synced', [
                'count' => count($templates),
                'phone_number_id' => $phoneNumberId
            ]);

            return [
                'success' => true,
                'count' => count($templates),
                'message' => 'Templates synced successfully'
            ];

        } catch (\Exception $e) {
            Log::error('Template sync failed: ' . $e->getMessage());
            
            CrmAuditLog::log('templates_sync_failed', [
                'error' => $e->getMessage()
            ]);

            return [
                'success' => false,
                'message' => 'Failed to sync templates: ' . $e->getMessage()
            ];
        }
    }

    public function sendTemplateMessage($to, $templateName, $language = 'en', $params = [])
    {
        try {
            $phoneNumberId = $this->getPhoneNumberId();
            $accessToken = $this->getAccessToken();

            if (!$phoneNumberId || !$accessToken) {
                throw new \Exception('Phone Number ID or Access Token not configured');
            }

            $payload = [
                'messaging_product' => 'whatsapp',
                'to' => $to,
                'type' => 'template',
                'template' => [
                    'name' => $templateName,
                    'language' => [
                        'code' => $language
                    ]
                ]
            ];

            // Add template parameters if provided
            if (!empty($params)) {
                $payload['template']['components'] = [];
                
                foreach ($params as $type => $components) {
                    $payload['template']['components'][] = [
                        'type' => $type,
                        'parameters' => $components
                    ];
                }
            }

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $accessToken,
                'Content-Type' => 'application/json'
            ])->post("{$this->baseUrl}/{$phoneNumberId}/messages", $payload);

            if (!$response->successful()) {
                $errorData = $response->json();
                throw new \Exception('Failed to send message: ' . ($errorData['error']['message'] ?? $response->body()));
            }

            $data = $response->json();
            
            CrmAuditLog::log('template_message_sent', [
                'to' => $to,
                'template' => $templateName,
                'message_id' => $data['messages'][0]['id'] ?? null
            ]);

            return [
                'success' => true,
                'message_id' => $data['messages'][0]['id'] ?? null,
                'data' => $data
            ];

        } catch (\Exception $e) {
            Log::error('Template message send failed: ' . $e->getMessage());
            
            CrmAuditLog::log('template_message_failed', [
                'to' => $to,
                'template' => $templateName,
                'error' => $e->getMessage()
            ]);

            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    public function sendTextMessage($to, $text)
    {
        try {
            $phoneNumberId = $this->getPhoneNumberId();
            $accessToken = $this->getAccessToken();

            if (!$phoneNumberId || !$accessToken) {
                throw new \Exception('Phone Number ID or Access Token not configured');
            }

            $payload = [
                'messaging_product' => 'whatsapp',
                'to' => $to,
                'type' => 'text',
                'text' => [
                    'body' => $text
                ]
            ];

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $accessToken,
                'Content-Type' => 'application/json'
            ])->post("{$this->baseUrl}/{$phoneNumberId}/messages", $payload);

            if (!$response->successful()) {
                $errorData = $response->json();
                throw new \Exception('Failed to send message: ' . ($errorData['error']['message'] ?? $response->body()));
            }

            $data = $response->json();
            
            CrmAuditLog::log('text_message_sent', [
                'to' => $to,
                'message_id' => $data['messages'][0]['id'] ?? null
            ]);

            return [
                'success' => true,
                'message_id' => $data['messages'][0]['id'] ?? null,
                'data' => $data
            ];

        } catch (\Exception $e) {
            Log::error('Text message send failed: ' . $e->getMessage());
            
            CrmAuditLog::log('text_message_failed', [
                'to' => $to,
                'error' => $e->getMessage()
            ]);

            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    public function verifyWebhook($mode, $token, $challenge)
    {
        $verifyToken = config('app.meta_verify_token', env('META_VERIFY_TOKEN'));
        
        if ($mode === 'subscribe' && $token === $verifyToken) {
            return $challenge;
        }
        
        return false;
    }
}
