<?php

namespace App\Jobs;

use App\Models\CrmMessage;
use App\Services\MetaService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class SendTemplateJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public $tries = 3;
    public $timeout = 30;

    protected $messageId;

    /**
     * Create a new job instance.
     */
    public function __construct($messageId)
    {
        $this->messageId = $messageId;
    }

    /**
     * Execute the job.
     */
    public function handle(MetaService $metaService): void
    {
        try {
            $message = CrmMessage::find($this->messageId);
            
            if (!$message) {
                Log::error("Message not found: {$this->messageId}");
                return;
            }

            if ($message->type === 'template') {
                $result = $metaService->sendTemplateMessage(
                    $message->customer_e164,
                    $message->template_name,
                    'en', // Default language
                    $message->params
                );
            } else {
                $result = $metaService->sendTextMessage(
                    $message->customer_e164,
                    $message->content
                );
            }

            if ($result['success']) {
                $message->update([
                    'status' => 'sent',
                    'provider_message_id' => $result['message_id'] ?? null
                ]);
                
                Log::info("Message sent successfully: {$this->messageId}");
            } else {
                $message->update([
                    'status' => 'failed',
                    'error_reason' => $result['error'] ?? 'Unknown error'
                ]);
                
                Log::error("Message send failed: {$this->messageId} - " . ($result['error'] ?? 'Unknown error'));
            }

        } catch (\Exception $e) {
            if ($message) {
                $message->update([
                    'status' => 'failed',
                    'error_reason' => $e->getMessage()
                ]);
            }
            
            Log::error("SendTemplateJob failed: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Handle a job failure.
     */
    public function failed(\Throwable $exception): void
    {
        $message = CrmMessage::find($this->messageId);
        
        if ($message) {
            $message->update([
                'status' => 'failed',
                'error_reason' => $exception->getMessage()
            ]);
        }
        
        Log::error("SendTemplateJob failed permanently: " . $exception->getMessage());
    }
}
