<?php

namespace App\Http\Controllers;

use App\Models\CrmMessage;
use App\Models\CrmAuditLog;
use App\Services\MetaService;
use App\Services\RoutingService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class WebhookController extends Controller
{
    protected $metaService;
    protected $routingService;

    public function __construct(MetaService $metaService, RoutingService $routingService)
    {
        $this->metaService = $metaService;
        $this->routingService = $routingService;
    }

    public function verify(Request $request)
    {
        $mode = $request->query('hub_mode');
        $token = $request->query('hub_verify_token');
        $challenge = $request->query('hub_challenge');

        $result = $this->metaService->verifyWebhook($mode, $token, $challenge);
        
        if ($result !== false) {
            return response($result, 200);
        }

        return response('Forbidden', 403);
    }

    public function handle(Request $request)
    {
        try {
            $data = $request->all();
            Log::info('Webhook received', $data);

            // Handle different types of webhook events
            if (isset($data['entry'][0]['changes'][0]['value']['messages'])) {
                $this->handleInboundMessage($data['entry'][0]['changes'][0]['value']);
            } elseif (isset($data['entry'][0]['changes'][0]['value']['statuses'])) {
                $this->handleStatusUpdate($data['entry'][0]['changes'][0]['value']);
            }

            return response('OK', 200);

        } catch (\Exception $e) {
            Log::error('Webhook processing failed: ' . $e->getMessage());
            return response('Error', 500);
        }
    }

    protected function handleInboundMessage($data)
    {
        $messages = $data['messages'] ?? [];
        
        foreach ($messages as $message) {
            $this->processInboundMessage($message);
        }
    }

    protected function processInboundMessage($message)
    {
        $customerE164 = $message['from'];
        $messageType = $message['type'] ?? 'text';
        $timestamp = $message['timestamp'] ?? time();
        $messageId = $message['id'] ?? null;

        // Determine message content based on type
        $content = '';
        if ($messageType === 'text') {
            $content = $message['text']['body'] ?? '';
        } elseif ($messageType === 'image') {
            $content = '[Image]';
        } elseif ($messageType === 'document') {
            $content = '[Document]';
        } else {
            $content = "[{$messageType}]";
        }

        // Get or assign agent for this customer
        $agent = $this->routingService->getAssignedAgent($customerE164);
        
        if (!$agent) {
            // Auto-assign customer to an agent
            $this->routingService->autoAssignCustomer($customerE164);
            $agent = $this->routingService->getAssignedAgent($customerE164);
        }

        // Create message record
        $crmMessage = CrmMessage::create([
            'customer_e164' => $customerE164,
            'agent_id' => $agent ? $agent->id : null,
            'direction' => 'inbound',
            'type' => $messageType,
            'content' => $content,
            'status' => 'delivered',
            'provider_message_id' => $messageId
        ]);

        CrmAuditLog::log('inbound_message_received', [
            'customer_e164' => $customerE164,
            'message_type' => $messageType,
            'agent_id' => $agent ? $agent->id : null
        ]);

        Log::info("Inbound message processed: {$customerE164} -> {$content}");
    }

    protected function handleStatusUpdate($data)
    {
        $statuses = $data['statuses'] ?? [];
        
        foreach ($statuses as $status) {
            $this->processStatusUpdate($status);
        }
    }

    protected function processStatusUpdate($status)
    {
        $messageId = $status['id'] ?? null;
        $statusType = $status['status'] ?? '';
        $timestamp = $status['timestamp'] ?? time();

        if (!$messageId) {
            return;
        }

        // Find the message by provider message ID
        $crmMessage = CrmMessage::where('provider_message_id', $messageId)->first();
        
        if (!$crmMessage) {
            Log::warning("Status update for unknown message: {$messageId}");
            return;
        }

        // Update message status
        $crmMessage->update(['status' => $statusType]);

        CrmAuditLog::log('message_status_updated', [
            'message_id' => $crmMessage->id,
            'provider_message_id' => $messageId,
            'status' => $statusType
        ]);

        Log::info("Message status updated: {$messageId} -> {$statusType}");
    }
}
