<?php

namespace App\Http\Controllers;

use App\Models\CrmMessage;
use App\Models\CrmTemplate;
use App\Models\CrmAuditLog;
use App\Jobs\SendTemplateJob;
use App\Services\MetaService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;

class MessagesController extends Controller
{
    protected $accountController;
    protected $metaService;

    public function __construct(AccountController $accountController, MetaService $metaService)
    {
        $this->accountController = $accountController;
        $this->metaService = $metaService;
    }

    public function compose()
    {
        $this->accountController->requireAuth();
        
        $templates = CrmTemplate::active()->get();
        return view('messages.compose', compact('templates'));
    }

    public function sendTemplate(Request $request)
    {
        $this->accountController->requireAuth();
        
        $request->validate([
            'to' => 'required|string',
            'template_name' => 'required|string|exists:crm_templates,name',
            'params' => 'nullable|array'
        ]);

        // Create message record
        $message = CrmMessage::create([
            'customer_e164' => $request->to,
            'agent_id' => Session::get('crm_user_id'),
            'direction' => 'outbound',
            'type' => 'template',
            'template_name' => $request->template_name,
            'params' => $request->params ?? [],
            'status' => 'queued'
        ]);

        // Dispatch job to send message
        SendTemplateJob::dispatch($message->id);

        CrmAuditLog::log('template_message_queued', [
            'to' => $request->to,
            'template' => $request->template_name,
            'queued_by' => Session::get('crm_username')
        ], Session::get('crm_user_id'));

        return back()->with('success', 'Message queued for sending.');
    }

    public function sendText(Request $request)
    {
        $this->accountController->requireAuth();
        
        $request->validate([
            'to' => 'required|string',
            'content' => 'required|string|max:1000'
        ]);

        // Create message record
        $message = CrmMessage::create([
            'customer_e164' => $request->to,
            'agent_id' => Session::get('crm_user_id'),
            'direction' => 'outbound',
            'type' => 'text',
            'content' => $request->input('content'),
            'status' => 'queued'
        ]);

        // Dispatch job to send message
        SendTemplateJob::dispatch($message->id);

        CrmAuditLog::log('text_message_queued', [
            'to' => $request->to,
            'queued_by' => Session::get('crm_username')
        ], Session::get('crm_user_id'));

        return back()->with('success', 'Message queued for sending.');
    }

    public function getTemplates()
    {
        $this->accountController->requireAuth();
        
        $templates = CrmTemplate::active()->get();
        return response()->json($templates);
    }

    public function getTemplateDetails($templateName)
    {
        $this->accountController->requireAuth();
        
        $template = CrmTemplate::where('name', $templateName)->first();
        
        if (!$template) {
            return response()->json(['error' => 'Template not found'], 404);
        }

        return response()->json($template);
    }

    public function messageHistory($customerE164)
    {
        $this->accountController->requireAuth();
        
        $messages = CrmMessage::forCustomer($customerE164)
                             ->with('agent')
                             ->orderBy('created_at', 'desc')
                             ->paginate(50);

        return view('messages.history', compact('messages', 'customerE164'));
    }

    public function getMessageStatus($messageId)
    {
        $this->accountController->requireAuth();
        
        $message = CrmMessage::findOrFail($messageId);
        
        return response()->json([
            'status' => $message->status,
            'error_reason' => $message->error_reason,
            'provider_message_id' => $message->provider_message_id
        ]);
    }

    public function resendMessage($messageId)
    {
        $this->accountController->requireAuth();
        
        $message = CrmMessage::findOrFail($messageId);
        
        if ($message->status !== 'failed') {
            return back()->with('error', 'Only failed messages can be resent.');
        }

        // Reset message status and dispatch job
        $message->update(['status' => 'queued', 'error_reason' => null]);
        SendTemplateJob::dispatch($message->id);

        CrmAuditLog::log('message_resent', [
            'message_id' => $messageId,
            'resent_by' => Session::get('crm_username')
        ], Session::get('crm_user_id'));

        return back()->with('success', 'Message queued for resending.');
    }
}
