<?php

namespace App\Http\Controllers;

use App\Models\CrmMessage;
use App\Models\CrmAgentAssignment;
use App\Models\CrmAuditLog;
use App\Services\RoutingService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;

class AgentController extends Controller
{
    protected $accountController;
    protected $routingService;

    public function __construct(AccountController $accountController, RoutingService $routingService)
    {
        $this->accountController = $accountController;
        $this->routingService = $routingService;
    }

    public function inbox()
    {
        $this->accountController->requireRole('Agent');
        
        $agentId = Session::get('crm_user_id');
        
        // Get customers assigned to this agent
        $assignments = CrmAgentAssignment::where('agent_id', $agentId)
                                        ->with(['messages' => function($query) {
                                            $query->latest();
                                        }])
                                        ->get();

        // Get recent conversations
        $conversations = [];
        foreach ($assignments as $assignment) {
            $lastMessage = $assignment->messages->first();
            if ($lastMessage) {
                $conversations[] = [
                    'customer_e164' => $assignment->customer_e164,
                    'last_message' => $lastMessage,
                    'unread_count' => $assignment->messages->where('direction', 'inbound')
                                                           ->where('status', 'delivered')
                                                           ->count()
                ];
            }
        }

        // Sort by last message time
        usort($conversations, function($a, $b) {
            return $b['last_message']->created_at <=> $a['last_message']->created_at;
        });

        return view('agent.inbox', compact('conversations'));
    }

    public function conversation($customerE164)
    {
        $this->accountController->requireRole('Agent');
        
        $agentId = Session::get('crm_user_id');
        
        // Verify this customer is assigned to this agent
        $assignment = CrmAgentAssignment::where('customer_e164', $customerE164)
                                       ->where('agent_id', $agentId)
                                       ->first();

        if (!$assignment) {
            abort(403, 'Customer not assigned to you.');
        }

        // Get conversation history
        $messages = CrmMessage::forCustomer($customerE164)
                             ->orderBy('created_at', 'asc')
                             ->get();

        // Mark inbound messages as read
        CrmMessage::forCustomer($customerE164)
                 ->inbound()
                 ->where('status', 'delivered')
                 ->update(['status' => 'read']);

        return view('agent.conversation', compact('messages', 'customerE164'));
    }

    public function sendMessage(Request $request, $customerE164)
    {
        $this->accountController->requireRole('Agent');
        
        $agentId = Session::get('crm_user_id');
        
        // Verify this customer is assigned to this agent
        $assignment = CrmAgentAssignment::where('customer_e164', $customerE164)
                                       ->where('agent_id', $agentId)
                                       ->first();

        if (!$assignment) {
            abort(403, 'Customer not assigned to you.');
        }

        $request->validate([
            'content' => 'required|string|max:1000'
        ]);

        // Create message record
        $message = CrmMessage::create([
            'customer_e164' => $customerE164,
            'agent_id' => $agentId,
            'direction' => 'outbound',
            'type' => 'text',
            'content' => $request->input('content'),
            'status' => 'queued'
        ]);

        // Dispatch job to send message
        \App\Jobs\SendTemplateJob::dispatch($message->id);

        CrmAuditLog::log('agent_message_sent', [
            'customer_e164' => $customerE164,
            'agent_id' => $agentId,
            'content' => $request->input('content')
        ], $agentId);

        return back()->with('success', 'Message sent.');
    }

    public function reassignCustomer(Request $request, $customerE164)
    {
        $this->accountController->requireRole('Agent');
        
        $request->validate([
            'new_agent_id' => 'required|exists:crm_users,id'
        ]);

        $this->routingService->reassignCustomer($customerE164, $request->new_agent_id);

        return back()->with('success', 'Customer reassigned successfully.');
    }

    public function dashboard()
    {
        $this->accountController->requireRole('Agent');
        
        $agentId = Session::get('crm_user_id');
        
        $stats = [
            'assigned_customers' => CrmAgentAssignment::where('agent_id', $agentId)->count(),
            'unread_messages' => CrmMessage::forAgent($agentId)
                                         ->inbound()
                                         ->where('status', 'delivered')
                                         ->count(),
            'messages_today' => CrmMessage::forAgent($agentId)
                                        ->whereDate('created_at', today())
                                        ->count(),
            'recent_activity' => CrmMessage::forAgent($agentId)
                                         ->latest()
                                         ->take(10)
                                         ->get()
        ];
        
        return view('agent.dashboard', compact('stats'));
    }

    public function getUnreadCount()
    {
        $this->accountController->requireRole('Agent');
        
        $agentId = Session::get('crm_user_id');
        
        $unreadCount = CrmMessage::forAgent($agentId)
                                ->inbound()
                                ->where('status', 'delivered')
                                ->count();
        
        return response()->json(['unread_count' => $unreadCount]);
    }

    public function markAsRead($customerE164)
    {
        $this->accountController->requireRole('Agent');
        
        $agentId = Session::get('crm_user_id');
        
        // Verify this customer is assigned to this agent
        $assignment = CrmAgentAssignment::where('customer_e164', $customerE164)
                                       ->where('agent_id', $agentId)
                                       ->first();

        if (!$assignment) {
            abort(403, 'Customer not assigned to you.');
        }

        // Mark all inbound messages as read
        CrmMessage::forCustomer($customerE164)
                 ->inbound()
                 ->where('status', 'delivered')
                 ->update(['status' => 'read']);

        return response()->json(['success' => true]);
    }
}
