<?php

namespace App\Http\Controllers;

use App\Models\CrmUser;
use App\Models\CrmSetting;
use App\Models\CrmTemplate;
use App\Models\CrmAuditLog;
use App\Services\MetaService;
use App\Jobs\SyncTemplatesJob;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;

class AdminController extends Controller
{
    protected $accountController;

    public function __construct(AccountController $accountController)
    {
        $this->accountController = $accountController;
    }

    public function settings()
    {
        $this->accountController->requireRole('Admin');
        
        $settings = CrmSetting::getSettings();
        $agents = CrmUser::where('role', 'Agent')->where('active', true)->get();
        $templates = CrmTemplate::active()->get();
        
        return view('admin.settings', compact('settings', 'agents', 'templates'));
    }

    public function updateSettings(Request $request)
    {
        $this->accountController->requireRole('Admin');
        
        $request->validate([
            'meta_access_token' => 'required|string',
            'phone_number_id' => 'required|string',
            'default_agent_id' => 'nullable|exists:crm_users,id'
        ]);

        $settings = CrmSetting::getSettings();
        $settings->updateSettings($request->only([
            'meta_access_token',
            'phone_number_id',
            'default_agent_id'
        ]));

        CrmAuditLog::log('settings_updated', [
            'updated_by' => Session::get('crm_username')
        ], Session::get('crm_user_id'));

        return back()->with('success', 'Settings updated successfully.');
    }

    public function syncTemplates()
    {
        $this->accountController->requireRole('Admin');
        
        SyncTemplatesJob::dispatch();

        CrmAuditLog::log('templates_sync_initiated', [
            'initiated_by' => Session::get('crm_username')
        ], Session::get('crm_user_id'));

        return back()->with('success', 'Template sync initiated. Check back in a few minutes.');
    }

    public function users()
    {
        $this->accountController->requireRole('Admin');
        
        $users = CrmUser::all();
        return view('admin.users', compact('users'));
    }

    public function createUser(Request $request)
    {
        $this->accountController->requireRole('Admin');
        
        $request->validate([
            'username' => 'required|string|unique:crm_users,username',
            'password' => 'required|string|min:6',
            'role' => 'required|in:Admin,Agent'
        ]);

        $user = CrmUser::create([
            'username' => $request->username,
            'password' => $request->password,
            'role' => $request->role,
            'active' => true
        ]);

        CrmAuditLog::log('user_created', [
            'created_user' => $user->username,
            'role' => $user->role,
            'created_by' => Session::get('crm_username')
        ], Session::get('crm_user_id'));

        return back()->with('success', 'User created successfully.');
    }

    public function updateUser(Request $request, $id)
    {
        $this->accountController->requireRole('Admin');
        
        $user = CrmUser::findOrFail($id);
        
        $request->validate([
            'username' => 'required|string|unique:crm_users,username,' . $id,
            'role' => 'required|in:Admin,Agent',
            'active' => 'boolean'
        ]);

        $user->update($request->only(['username', 'role', 'active']));

        if ($request->filled('password')) {
            $user->update(['password' => $request->password]);
        }

        CrmAuditLog::log('user_updated', [
            'updated_user' => $user->username,
            'updated_by' => Session::get('crm_username')
        ], Session::get('crm_user_id'));

        return back()->with('success', 'User updated successfully.');
    }

    public function deleteUser($id)
    {
        $this->accountController->requireRole('Admin');
        
        $user = CrmUser::findOrFail($id);
        
        // Prevent deleting the last admin
        if ($user->role === 'Admin' && CrmUser::where('role', 'Admin')->count() <= 1) {
            return back()->with('error', 'Cannot delete the last admin user.');
        }

        $username = $user->username;
        $user->delete();

        CrmAuditLog::log('user_deleted', [
            'deleted_user' => $username,
            'deleted_by' => Session::get('crm_username')
        ], Session::get('crm_user_id'));

        return back()->with('success', 'User deleted successfully.');
    }

    public function dashboard()
    {
        $this->accountController->requireRole('Admin');
        
        $stats = [
            'total_users' => CrmUser::count(),
            'active_agents' => CrmUser::where('role', 'Agent')->where('active', true)->count(),
            'total_templates' => CrmTemplate::active()->count(),
            'recent_audit_logs' => CrmAuditLog::latest()->take(10)->get()
        ];
        
        return view('admin.dashboard', compact('stats'));
    }
}
