<?php

namespace App\Http\Controllers;

use App\Models\CrmUser;
use App\Models\CrmAuditLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Session;

class AccountController extends Controller
{
    public function showLogin()
    {
        if (Session::has('crm_user_id')) {
            $user = CrmUser::find(Session::get('crm_user_id'));
            if ($user && $user->active) {
                return $this->redirectBasedOnRole($user);
            }
        }
        
        return view('auth.login');
    }

    public function login(Request $request)
    {
        $request->validate([
            'username' => 'required|string',
            'password' => 'required|string',
        ]);

        $user = CrmUser::where('username', $request->username)
                      ->where('active', true)
                      ->first();

        if (!$user || !Hash::check($request->password, $user->password)) {
            return back()->withErrors([
                'username' => 'Invalid credentials.',
            ])->withInput($request->only('username'));
        }

        Session::put('crm_user_id', $user->id);
        Session::put('crm_user_role', $user->role);
        Session::put('crm_username', $user->username);

        CrmAuditLog::log('user_login', [
            'username' => $user->username,
            'role' => $user->role
        ], $user->id);

        return $this->redirectBasedOnRole($user);
    }

    public function logout()
    {
        $userId = Session::get('crm_user_id');
        
        if ($userId) {
            CrmAuditLog::log('user_logout', [
                'username' => Session::get('crm_username')
            ], $userId);
        }

        Session::forget(['crm_user_id', 'crm_user_role', 'crm_username']);
        
        return redirect('/login')->with('success', 'Logged out successfully.');
    }

    protected function redirectBasedOnRole($user)
    {
        if ($user->isAdmin()) {
            return redirect('/admin/settings');
        } else {
            return redirect('/agent/inbox');
        }
    }

    public function getCurrentUser()
    {
        $userId = Session::get('crm_user_id');
        return $userId ? CrmUser::find($userId) : null;
    }

    public function isAuthenticated()
    {
        return Session::has('crm_user_id');
    }

    public function requireAuth()
    {
        if (!$this->isAuthenticated()) {
            return redirect('/login');
        }
    }

    public function requireRole($role)
    {
        $this->requireAuth();
        
        $userRole = Session::get('crm_user_role');
        if ($userRole !== $role) {
            abort(403, 'Unauthorized access.');
        }
    }
}
