<?php
header('Content-Type: application/json; charset=utf-8');
require_once __DIR__ . '/vapi_config.php';

// --- Auth (shared-secret header) ---
$got = $_SERVER['HTTP_X_VAPI_SECRET'] ?? '';
if (!$got || !hash_equals(VAPI_WEBHOOK_SECRET, $got)) {
    http_response_code(401);
    echo json_encode(['error' => 'unauthorized']);
    exit;
}

// --- Read JSON ---
$raw = file_get_contents('php://input');
$payload = json_decode($raw, true) ?: [];
$msg = $payload['message'] ?? $payload;
$type = strtolower($msg['type'] ?? '');

// --- DB Connection ---
try {
    $pdo = new PDO(DB_DSN, DB_USER, DB_PASS, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4"
    ]);
} catch (Exception $e) {
    error_log('DB Connection failed in webhook: ' . $e->getMessage());
    echo json_encode(['ok' => false, 'error' => 'db_connection_failed']);
    exit;
}

// --- Extract basic call info ---
$call = $msg['call'] ?? [];
$header_call_id = $_SERVER['HTTP_X_CALL_ID'] ?? null; // prefer header if present
$call_id = $header_call_id ?: ($call['id'] ?? ($call['callId'] ?? bin2hex(random_bytes(16))));

// --- Handle different webhook events ---
try {
    switch($type) {
        case 'end-of-call-report':
        case 'conversation-update':
        case 'status-update':
            handleCallData($pdo, $msg, $call_id, $raw, $type);
            
            // Trigger auto-analysis for end-of-call reports
            if ($type === 'end-of-call-report') {
                triggerAutoAnalysis($call_id);
            }
            break;

        case 'transcript':
        case 'transcript-updated':
            handleTranscript($pdo, $msg, $call_id);
            break;

        case 'recording-available':
            handleRecording($pdo, $msg, $call_id);
            break;

        // capture numbers coming from tools (e.g., notes tool)
        case 'tool-calls':
        case 'tool.completed':
        case 'tool-calls-result':
            handleToolPhones($pdo, $msg, $call_id, $raw);
            echo json_encode(['ok' => true, 'type' => $type, 'captured' => true]);
            exit;

        case 'speech-update':
        case 'function-call':
            echo json_encode(['ok' => true, 'type' => $type, 'acknowledged' => true]);
            exit;

        default:
            echo json_encode(['ok' => true, 'skipped' => $type]);
            exit;
    }

    echo json_encode(['ok' => true, 'processed' => $type]);

} catch (Throwable $e) {
    error_log('VAPI webhook error: ' . $e->getMessage() . ' | Type: ' . $type . ' | Call ID: ' . $call_id);
    echo json_encode(['ok' => false, 'error' => 'logged']);
}

/* =========================
   Auto Analysis Functions - MODIFIED TO ANALYZE ALL CALLS
   ========================= */

function triggerAutoAnalysis($call_id) {
    // MODIFIED: Always try to analyze, even without transcript
    error_log("[auto-analysis] Triggering analysis for call: $call_id");
    
    // Trigger analysis in background (non-blocking)
    $analysis_url = getAnalysisUrl();
    
    if (!$analysis_url) {
        error_log("[auto-analysis] No analysis URL configured");
        return;
    }
    
    // Prepare data for analysis
    $post_data = json_encode([
        'action' => 'analyze_single',
        'call_id' => $call_id,
        'trigger' => 'webhook_auto'
    ]);
    
    // Use async request to avoid blocking webhook response
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $analysis_url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $post_data);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'X-Webhook-Trigger: auto-analysis'
    ]);
    curl_setopt($ch, CURLOPT_TIMEOUT, 3); // Quick timeout to avoid blocking
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_NOSIGNAL, 1);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, false);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($response && $http_code == 200) {
        error_log("[auto-analysis] Successfully triggered analysis for call: $call_id");
    } else {
        error_log("[auto-analysis] Failed to trigger analysis for call: $call_id (HTTP: $http_code)");
    }
}

// MODIFIED: Removed transcript requirements - analyze ALL calls
function hasTranscriptForAnalysis($call_id) {
    // MODIFIED: Always return true to analyze every call
    return true;
    
    // OLD CODE (commented out):
    /*
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            SELECT COUNT(*) as message_count,
                   SUM(CHAR_LENGTH(message)) as total_chars
            FROM vapi_call_transcripts 
            WHERE call_id = ? AND message IS NOT NULL AND message != ''
        ");
        $stmt->execute([$call_id]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // OLD: Require at least 3 messages and 100 characters
        return ($result['message_count'] >= 3 && $result['total_chars'] >= 100);
        
    } catch (Exception $e) {
        error_log("[auto-analysis] Error checking transcript for $call_id: " . $e->getMessage());
        return false;
    }
    */
}

function getAnalysisUrl() {
    // Determine the analysis URL based on current domain
    $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https://' : 'http://';
    $domain = $_SERVER['HTTP_HOST'] ?? 'localhost';
    
    // Try to construct the URL to call_analyzer.php
    return $protocol . $domain . dirname($_SERVER['SCRIPT_NAME']) . '/call_analyzer.php';
}

/* =========================
   Helpers
   ========================= */

function normalizePhone($s) {
    if (!$s || !is_string($s)) return null;

    // For SIP URIs, extract a plausible phone token
    if (stripos($s, 'sip:') !== false || strpos($s, '@') !== false) {
        if (preg_match('/\+?\d{9,15}/', $s, $m)) {
            $t = $m[0];
            if ($t[0] === '+' || $t[0] === '0') return $t;
            return null;
        }
        return null;
    }

    if (preg_match('/\+?\d{9,15}/', $s, $m)) {
        $t = $m[0];
        if ($t[0] === '+' || $t[0] === '0') return $t;
        return null;
    }
    return null;
}

// Prefer explicit PSTN/SIP fields; do not deep-scan for web calls.
function extractPhonesFromCall($msg, $call) {
    $dir  = strtolower($call['type'] ?? $call['direction'] ?? ($msg['direction'] ?? ''));

    $from = $call['fromNumber'] ?? $call['from'] ??
            ($call['transport']['from'] ?? $call['transport']['sipFrom'] ?? null);
    $to   = $call['toNumber']   ?? $call['to']   ??
            ($call['transport']['to']   ?? $call['transport']['sipTo']   ?? null);

    // Top-level fallbacks
    $from = $from ?? ($msg['fromNumber'] ?? $msg['from'] ?? null);
    $to   = $to   ?? ($msg['toNumber']   ?? $msg['to']   ?? null);

    // Header fallbacks some providers set
    $from = $from ?? ($_SERVER['HTTP_X_FROM_NUMBER'] ?? null);
    $to   = $to   ?? ($_SERVER['HTTP_X_TO_NUMBER'] ?? null);

    if ($dir === 'inbound') {
        $customer  = $from;
        $assistant = $to;
    } else {
        $assistant = $from;
        $customer  = $to;
    }

    return [
        'assistant' => normalizePhone($assistant),
        'customer'  => normalizePhone($customer),
        'direction' => $dir ?: null,
        'raw_from'  => $from,
        'raw_to'    => $to,
    ];
}

// Collect all places Vapi might put variables/variableValues
function collectVariableScopes($msg, $call, $artifact) {
    $scopes = [];
    foreach ([
        $msg['variableValues'] ?? null,
        $call['assistantOverrides']['variableValues'] ?? null,
        $call['variableValues'] ?? null,
        $msg['variables'] ?? null,
        $call['variables'] ?? null,
        $artifact['variables'] ?? null,
        $msg['conversation']['variables'] ?? null,
    ] as $cand) {
        if (is_array($cand) && !empty($cand)) $scopes[] = $cand;
    }
    return $scopes;
}

// Find assistant DID under ...phoneNumber.number (recursively if needed)
function findAssistantDidInVars($node) {
    if (!is_array($node)) return null;
    if (isset($node['phoneNumber']['number']) && is_string($node['phoneNumber']['number'])) {
        return $node['phoneNumber']['number'];
    }
    foreach ($node as $v) {
        if (is_array($v)) {
            $hit = findAssistantDidInVars($v);
            if ($hit) return $hit;
        }
    }
    return null;
}

/* =========================
   Core handlers
   ========================= */

function handleCallData($pdo, $msg, $call_id, $raw, $event_type) {
    $call     = $msg['call'] ?? [];
    $meta     = $call['meta'] ?? [];
    $metrics  = $msg['metrics'] ?? [];
    $artifact = $msg['artifact'] ?? [];
    $costs    = $msg['costs'] ?? [];

    // Check if record already exists
    $existing = $pdo->prepare("SELECT * FROM vapi_call_logs WHERE call_id = ?");
    $existing->execute([$call_id]);
    $existingData = $existing->fetch(PDO::FETCH_ASSOC);

    // Extract new data
    $newData = extractCallData($msg, $call, $meta, $metrics, $artifact, $costs);

    if ($existingData) {
        updateExistingRecord($pdo, $call_id, $existingData, $newData, $raw);
    } else {
        insertNewRecord($pdo, $call_id, $newData, $raw);
    }

    handleTranscriptIfAvailable($pdo, $msg, $call_id);
    handleRecordingIfAvailable($pdo, $msg, $call_id);
}

function extractCallData($msg, $call, $meta, $metrics, $artifact, $costs) {
    $data = [];

    // Assistant info (prefer message.assistant.name)
    $data['assistant_name'] =
        $msg['assistant']['name']
        ?? ($meta['assistantName'] ?? ($meta['configName'] ?? ($call['assistantName'] ?? 'maharah')));

    // ------------ Phones ------------
    $phones = extractPhonesFromCall($msg, $call);

    // Gather BOTH variableValues and variables (Vapi may use either)
    $vv_sources = collectVariableScopes($msg, $call, $artifact);

    $assistant_did = null; // e.g., +966115001900
    $customer_cli  = null; // e.g., 0501049130

    foreach ($vv_sources as $vv) {
        if (!$assistant_did) {
            $assistant_did = findAssistantDidInVars($vv);
        }
        if (!$customer_cli && isset($vv['phone'])) {
            $customer_cli = $vv['phone'];
        }
        if (!$customer_cli && isset($vv['customer']['sipUri'])) {
            $customer_cli = normalizePhone($vv['customer']['sipUri']);
        }
    }

    // Map raw endpoints according to direction
    if (strtolower($phones['direction'] ?? '') === 'inbound') {
        $assistant_raw = $phones['raw_to']   ?? ($call['toNumber']   ?? $call['to']   ?? null);
        $customer_raw  = $phones['raw_from'] ?? ($call['fromNumber'] ?? $call['from'] ?? null);
    } else {
        $assistant_raw = $phones['raw_from'] ?? ($call['fromNumber'] ?? $call['from'] ?? null);
        $customer_raw  = $phones['raw_to']   ?? ($call['toNumber']   ?? $call['to']   ?? null);
    }

    // Final values: prefer variables.* (what Vapi UI shows), then raw endpoints, then normalized
    $data['assistant_phone'] = $assistant_did
        ?? $assistant_raw
        ?? ($phones['assistant'] ?? $call['assistantPhoneNumber'] ?? $meta['assistantPhone'] ?? $call['assistantPhone'] ?? null);

    $data['customer_phone']  = $customer_cli
        ?? $customer_raw
        ?? ($phones['customer'] ?? $call['customerPhoneNumber'] ?? null);

    // Debug (remove later)
    error_log("[vars] did=" . json_encode($assistant_did) . " cli=" . json_encode($customer_cli));
    error_log("[phones] dir={$phones['direction']} raw_from=" . json_encode($phones['raw_from']) .
              " raw_to=" . json_encode($phones['raw_to']) .
              " assistant_final=" . json_encode($data['assistant_phone']) .
              " customer_final=" . json_encode($data['customer_phone']));

    // Fallback from notes tool default if present and real
    if (!$data['customer_phone'] && isset($msg['assistant']['tools'])) {
        foreach ($msg['assistant']['tools'] as $tool) {
            if (($tool['name'] ?? '') === 'notes'
                && isset($tool['body']['properties']['caller_number']['default'])) {
                $v = $tool['body']['properties']['caller_number']['default'];
                if ($v !== '{{phone}}') {
                    $data['customer_phone'] = $v; // store as-is to match UI
                    break;
                }
            }
        }
    }

    // --- Call type
    $call_direction = $phones['direction'] ?: ($call['type'] ?? $call['direction'] ?? '');
    $data['call_type'] = (strtolower($call_direction) === 'inbound') ? 'Inbound' : 'Outbound';

    // --- Call outcome
    $data['ended_reason'] = $msg['endedReason'] ??
                            ($metrics['endedReason'] ?? ($call['endedReason'] ?? null));

    // --- Success evaluation
    $data['success_eval'] = null;
    if (isset($msg['analysis']['successEvaluation'])) {
        $v = $msg['analysis']['successEvaluation'];
        $data['success_eval'] = is_bool($v) ? ($v ? 'Success' : 'Fail')
                          : ((is_string($v) && strtolower($v)==='true') ? 'Success'
                          : (is_string($v) ? 'Fail' : null));
    }
    if (!$data['success_eval']) {
        $data['success_eval'] = $msg['successEvaluation'] ??
                               ($metrics['successEvaluation'] ??
                               (isset($msg['success']) ? ($msg['success'] ? 'Success' : 'Fail') : 'Success'));
    }

    // --- Timing (UTC → Asia/Riyadh)
    $started_iso = $call['startedAt'] ?? ($artifact['startedAt'] ?? ($msg['startedAt'] ?? null));
    $data['start_time'] = null;
    if ($started_iso) {
        try {
            $utc = new DateTime($started_iso, new DateTimeZone('UTC'));
            $utc->setTimezone(new DateTimeZone('Asia/Riyadh'));
            $data['start_time'] = $utc->format('Y-m-d H:i:s');
        } catch (Exception $e) {
            $data['start_time'] = null;
        }
    }

    // --- Duration
    $data['duration_seconds'] = $metrics['durationSeconds'] ??
                                ($metrics['talkSeconds'] ??
                                ($call['durationSeconds'] ?? ($msg['durationSeconds'] ?? null)));

    // --- Cost
    $data['cost_usd'] = null;
    if (isset($msg['cost']) && is_numeric($msg['cost'])) {
        $data['cost_usd'] = floatval($msg['cost']);
    } elseif (isset($msg['costs']) && is_array($msg['costs'])) {
        $sum = 0.0;
        foreach ($msg['costs'] as $c) {
            if (isset($c['cost']) && is_numeric($c['cost'])) $sum += floatval($c['cost']);
        }
        if ($sum > 0) $data['cost_usd'] = $sum;
    } elseif (is_array($costs) && !empty($costs)) {
        $sum = 0.0;
        foreach ($costs as $c) {
            if (is_array($c)) $sum += floatval($c['quantity'] ?? 0) * floatval($c['unitPrice'] ?? 0);
        }
        $data['cost_usd'] = $sum > 0 ? $sum : null;
    } elseif (is_numeric($costs)) {
        $data['cost_usd'] = floatval($costs);
    }

    return $data;
}

function insertNewRecord($pdo, $call_id, $data, $raw) {
    $sql = "INSERT INTO vapi_call_logs
      (call_id, assistant_name, assistant_phone, customer_phone, call_type,
       ended_reason, success_eval, start_time, duration_seconds, cost_usd, raw_json)
     VALUES
      (:call_id, :assistant_name, :assistant_phone, :customer_phone, :call_type,
       :ended_reason, :success_eval, :start_time, :duration_seconds, :cost_usd, :raw_json)";

    $stmt = $pdo->prepare($sql);
    $stmt->execute([
        ':call_id' => $call_id,
        ':assistant_name' => $data['assistant_name'],
        ':assistant_phone' => $data['assistant_phone'],
        ':customer_phone' => $data['customer_phone'],
        ':call_type' => $data['call_type'],
        ':ended_reason' => $data['ended_reason'],
        ':success_eval' => $data['success_eval'],
        ':start_time' => $data['start_time'],
        ':duration_seconds' => $data['duration_seconds'],
        ':cost_usd' => $data['cost_usd'],
        ':raw_json' => $raw
    ]);
}

function updateExistingRecord($pdo, $call_id, $existing, $new, $raw) {
    $updates = [];
    $params = [':call_id' => $call_id];

    foreach ($new as $field => $newValue) {
        $existingValue = $existing[$field];
        if (($existingValue === null && $newValue !== null) ||
            ($newValue !== null && $newValue !== $existingValue)) {
            $updates[] = "$field = :$field";
            $params[":$field"] = $newValue;
        }
    }

    // Always update raw_json and timestamp
    $updates[] = "raw_json = :raw_json";
    $params[':raw_json'] = $raw;
    $updates[] = "updated_at = CURRENT_TIMESTAMP";

    if (!empty($updates)) {
        $sql = "UPDATE vapi_call_logs SET " . implode(', ', $updates) . " WHERE call_id = :call_id";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
    }
}

// Capture numbers that arrive via tool calls (e.g., notes tool asking the customer)
function handleToolPhones($pdo, $msg, $call_id, $raw) {
    $candidates = [];
    foreach (['tools','toolCalls','tool','body','args','payload','parameters'] as $k) {
        if (isset($msg[$k])) $candidates[] = $msg[$k];
    }
    $candidates[] = $msg; // scan whole message as fallback

    $hits = [];
    $stack = $candidates;
    while ($stack) {
        $cur = array_pop($stack);
        if (is_array($cur)) {
            foreach ($cur as $v) {
                if (is_array($v)) { $stack[] = $v; continue; }
                if (is_string($v)) {
                    $n = normalizePhone($v);
                    if ($n) $hits[] = $n;
                }
            }
        } elseif (is_string($cur)) {
            $n = normalizePhone($cur);
            if ($n) $hits[] = $n;
        }
    }

    if (!$hits) return;
    $customer = $hits[0]; // choose the first plausible number

    $stmt = $pdo->prepare("UPDATE vapi_call_logs
                           SET customer_phone = COALESCE(customer_phone, :phone),
                               raw_json = :raw,
                               updated_at = CURRENT_TIMESTAMP
                           WHERE call_id = :call_id");
    $stmt->execute([
        ':phone' => $customer,
        ':raw'   => $raw,
        ':call_id' => $call_id
    ]);

    error_log("[tool-phone] call={$call_id} chosen={$customer} all=" . json_encode($hits));
}

/* =========================
   Transcript / Recording
   ========================= */

function handleTranscriptIfAvailable($pdo, $msg, $call_id) {
    $transcript = $msg['transcript'] ??
                  ($msg['call']['transcript'] ??
                  ($msg['conversation']['messages'] ??
                  ($msg['artifact']['messages'] ?? [])));
    if (is_array($transcript) && !empty($transcript)) {
        handleTranscript($pdo, $msg, $call_id);
    }
}

function handleRecordingIfAvailable($pdo, $msg, $call_id) {
    $recording_url = $msg['recordingUrl'] ??
                     ($msg['recording_url'] ??
                     ($msg['call']['recordingUrl'] ??
                     ($msg['artifact']['recordingUrl'] ?? null)));
    if ($recording_url) {
        handleRecording($pdo, $msg, $call_id);
    }
}

function handleTranscript($pdo, $msg, $call_id) {
    $transcript = $msg['transcript'] ??
                  ($msg['call']['transcript'] ??
                  ($msg['conversation']['messages'] ??
                  ($msg['artifact']['messages'] ?? [])));
    if (!is_array($transcript) || empty($transcript)) return;

    // Clear existing transcript for this call
    $pdo->prepare("DELETE FROM vapi_call_transcripts WHERE call_id = ?")->execute([$call_id]);

    // Insert new transcript entries
    $stmt = $pdo->prepare("
        INSERT INTO vapi_call_transcripts (call_id, role, message, timestamp, sequence_number)
        VALUES (?, ?, ?, ?, ?)
    ");

    foreach ($transcript as $index => $entry) {
        if (!is_array($entry)) continue;

        $role = strtolower($entry['role'] ?? 'unknown');
        if (in_array($role, ['bot', 'assistant', 'ai', 'system'])) $role = 'assistant';
        elseif (in_array($role, ['human', 'customer', 'caller', 'user'])) $role = 'user';

        $message = $entry['message'] ?? ($entry['text'] ?? ($entry['content'] ?? ''));
        if (empty($message)) continue;

        $timestamp = null;
        if (isset($entry['timestamp'])) {
            try { $timestamp = date('Y-m-d H:i:s', strtotime($entry['timestamp'])); }
            catch (Exception $e) { $timestamp = null; }
        }

        $stmt->execute([$call_id, $role, $message, $timestamp, $index]);
    }
}

function handleRecording($pdo, $msg, $call_id) {
    $recording_url = $msg['recordingUrl'] ??
                     ($msg['recording_url'] ??
                     ($msg['call']['recordingUrl'] ??
                     ($msg['artifact']['recordingUrl'] ?? null)));
    if (!$recording_url) return;

    try {
        $stmt = $pdo->prepare("
            INSERT INTO vapi_call_recordings (call_id, recording_url)
            VALUES (?, ?)
            ON DUPLICATE KEY UPDATE
                recording_url = VALUES(recording_url),
                download_status = 'pending'
        ");
        $stmt->execute([$call_id, $recording_url]);
    } catch (Exception $e) {
        error_log('Error saving recording URL: ' . $e->getMessage());
    }
}
?>