<?php
// vapi_api.php - FIXED VERSION with proper timeout and error handling
require_once __DIR__ . '/vapi_config.php';

header('Content-Type: application/json; charset=utf-8');

class VapiManager {
    private $api_key;
    private $base_url = 'https://api.vapi.ai';
    
    public function __construct($api_key) {
        $this->api_key = $api_key;
    }
    
    private function apiRequest($method, $endpoint, $data = null) {
        $url = $this->base_url . $endpoint;
        
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_CUSTOMREQUEST => $method,
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $this->api_key,
                'Content-Type: application/json'
            ],
            CURLOPT_TIMEOUT => 15,  // FIXED: Reduced from 30 to 15 seconds
            CURLOPT_CONNECTTIMEOUT => 10,  // ADDED: Connection timeout
            CURLOPT_FOLLOWLOCATION => true,  // ADDED: Follow redirects
            CURLOPT_MAXREDIRS => 3,  // ADDED: Max 3 redirects
            CURLOPT_SSL_VERIFYPEER => true,  // ADDED: Verify SSL
            CURLOPT_SSL_VERIFYHOST => 2  // ADDED: Verify host
        ]);
        
        if ($data !== null && in_array($method, ['POST', 'PATCH', 'PUT'])) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        }
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curl_error = curl_error($ch);
        $curl_errno = curl_errno($ch);
        curl_close($ch);
        
        // ADDED: Better error logging
        if ($curl_error) {
            error_log("VAPI API Error: $curl_error (Code: $curl_errno) | Endpoint: $endpoint");
            return [
                'error' => true, 
                'message' => 'Connection error: ' . $curl_error,
                'errno' => $curl_errno,
                'endpoint' => $endpoint
            ];
        }
        
        // ADDED: Handle empty response
        if (empty($response)) {
            error_log("VAPI API: Empty response | HTTP Code: $http_code | Endpoint: $endpoint");
            return [
                'error' => true,
                'http_code' => $http_code,
                'message' => 'Empty response from API',
                'endpoint' => $endpoint
            ];
        }
        
        if ($http_code >= 400) {
            error_log("VAPI API Error: HTTP $http_code | Response: $response | Endpoint: $endpoint");
            return [
                'error' => true,
                'http_code' => $http_code,
                'message' => $response,
                'response' => json_decode($response, true),
                'endpoint' => $endpoint
            ];
        }
        
        $decoded = json_decode($response, true);
        
        // ADDED: Validate JSON response
        if (json_last_error() !== JSON_ERROR_NONE) {
            error_log("VAPI API: Invalid JSON response | Error: " . json_last_error_msg() . " | Response: " . substr($response, 0, 200));
            return [
                'error' => true,
                'message' => 'Invalid JSON response: ' . json_last_error_msg(),
                'raw_response' => substr($response, 0, 500)
            ];
        }
        
        return $decoded;
    }
    
    public function getAssistants($limit = 100) {
        return $this->apiRequest('GET', '/assistant?limit=' . $limit);
    }
    
    public function getAssistant($assistant_id) {
        return $this->apiRequest('GET', '/assistant/' . $assistant_id);
    }
    
    public function createAssistant($config) {
        return $this->apiRequest('POST', '/assistant', $config);
    }
    
    public function updateAssistant($assistant_id, $updates) {
        return $this->apiRequest('PATCH', '/assistant/' . $assistant_id, $updates);
    }
    
    public function deleteAssistant($assistant_id) {
        return $this->apiRequest('DELETE', '/assistant/' . $assistant_id);
    }
    
    public function getCalls($limit = 100) {
        return $this->apiRequest('GET', '/call?limit=' . $limit);
    }
    
    public function getCall($call_id) {
        return $this->apiRequest('GET', '/call/' . $call_id);
    }
    
    public function getSquads($limit = 100) {
        return $this->apiRequest('GET', '/squad?limit=' . $limit);
    }
    
    public function getPhoneNumbers($limit = 100) {
        return $this->apiRequest('GET', '/phone-number?limit=' . $limit);
    }
    
    public function getTools($limit = 100) {
        return $this->apiRequest('GET', '/tool?limit=' . $limit);
    }
}

// ADDED: Better API key validation
if (!defined('VAPI_API_KEY')) {
    error_log('VAPI_API_KEY not defined in vapi_config.php');
    http_response_code(500);
    die(json_encode(['error' => true, 'message' => 'VAPI_API_KEY not configured']));
}

if (empty(VAPI_API_KEY)) {
    error_log('VAPI_API_KEY is empty');
    http_response_code(500);
    die(json_encode(['error' => true, 'message' => 'VAPI_API_KEY is empty']));
}

$vapi = new VapiManager(VAPI_API_KEY);
$input = json_decode(file_get_contents('php://input'), true);
$action = $input['action'] ?? '';

// ADDED: Log incoming requests for debugging
error_log("VAPI API Request: Action=$action | Input=" . json_encode($input));

try {
    switch($action) {
        case 'list_assistants':
            $result = $vapi->getAssistants($input['limit'] ?? 100);
            echo json_encode($result);
            break;
            
        case 'get_assistant':
            if (!isset($input['assistant_id'])) throw new Exception('Missing assistant_id');
            echo json_encode($vapi->getAssistant($input['assistant_id']));
            break;
            
        case 'create_assistant':
            if (!isset($input['config'])) throw new Exception('Missing config');
            echo json_encode($vapi->createAssistant($input['config']));
            break;
            
        case 'update_assistant':
            if (!isset($input['assistant_id'])) throw new Exception('Missing assistant_id');
            echo json_encode($vapi->updateAssistant($input['assistant_id'], $input['updates'] ?? []));
            break;
            
        case 'delete_assistant':
            if (!isset($input['assistant_id'])) throw new Exception('Missing assistant_id');
            echo json_encode($vapi->deleteAssistant($input['assistant_id']));
            break;
            
        case 'list_calls':
            echo json_encode($vapi->getCalls($input['limit'] ?? 100));
            break;
            
        case 'get_call':
            if (!isset($input['call_id'])) throw new Exception('Missing call_id');
            echo json_encode($vapi->getCall($input['call_id']));
            break;
            
        case 'list_squads':
            echo json_encode($vapi->getSquads());
            break;
            
        case 'list_phone_numbers':
            echo json_encode($vapi->getPhoneNumbers());
            break;
            
        case 'list_tools':
            echo json_encode($vapi->getTools());
            break;
            
        default:
            throw new Exception('Unknown action: ' . $action);
    }
} catch (Exception $e) {
    error_log("VAPI API Exception: " . $e->getMessage());
    http_response_code(400);
    echo json_encode(['error' => true, 'message' => $e->getMessage()]);
}