<?php
// Enhanced Manager Dashboard - Real Data from Databases + Call Logs Stats
require_once __DIR__ . '/vapi_config.php';

session_start();
if (!isset($_SESSION['logged_in'])) {
    header('Location: call_logs.php');
    exit;
}

// Helper functions
function h($s) { return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
function formatDuration($seconds) {
    if (!$seconds) return '0s';
    $hours = intdiv($seconds, 3600);
    $minutes = intdiv($seconds % 3600, 60);
    $secs = $seconds % 60;
    
    if ($hours > 0) return $hours . 'h ' . $minutes . 'm';
    if ($minutes > 0) return $minutes . 'm ' . $secs . 's';
    return $secs . 's';
}
function formatMoney($amount) {
    return '$' . number_format((float)$amount, 2);
}

// Date range for dashboard (default last 30 days)
$date_from = $_GET['date_from'] ?? date('Y-m-d', strtotime('-30 days'));
$date_to = $_GET['date_to'] ?? date('Y-m-d');

try {
    $pdo = new PDO(DB_DSN, DB_USER, DB_PASS, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4"
    ]);
} catch (Exception $e) {
    die("Database connection failed");
}

// 1. OVERALL CALL LOGS STATISTICS (Raw Call Data)
$call_logs_stats = $pdo->prepare("
    SELECT 
        COUNT(*) as total_calls,
        COUNT(CASE WHEN success_eval = 'Success' THEN 1 END) as successful_calls,
        COUNT(CASE WHEN success_eval = 'Fail' THEN 1 END) as failed_calls,
        SUM(duration_seconds) as total_duration,
        SUM(cost_usd) as total_cost,
        AVG(duration_seconds) as avg_duration,
        AVG(cost_usd) as avg_cost,
        COUNT(DISTINCT assistant_name) as active_agents,
        COUNT(DISTINCT customer_phone) as unique_customers,
        COUNT(CASE WHEN call_type = 'Inbound' THEN 1 END) as inbound_calls,
        COUNT(CASE WHEN call_type = 'Outbound' THEN 1 END) as outbound_calls,
        MIN(start_time) as first_call_date,
        MAX(start_time) as last_call_date,
        COUNT(CASE WHEN duration_seconds < 30 THEN 1 END) as very_short_calls,
        COUNT(CASE WHEN duration_seconds > 300 THEN 1 END) as long_calls
    FROM vapi_call_logs 
    WHERE start_time BETWEEN ? AND ?
");
$call_logs_stats->execute([$date_from . ' 00:00:00', $date_to . ' 23:59:59']);
$call_stats = $call_logs_stats->fetch(PDO::FETCH_ASSOC);

// 2. ANALYSIS STATISTICS (من جدول التحليل)
$analysis_stats = $pdo->prepare("
    SELECT 
        COUNT(*) as analyzed_calls,
        COUNT(CASE WHEN customer_satisfaction IN ('راضي جداً', 'راضي') THEN 1 END) as satisfied_customers,
        COUNT(CASE WHEN call_result = 'نجح' THEN 1 END) as successful_outcomes,
        COUNT(CASE WHEN should_call_again = 1 THEN 1 END) as needs_followup,
        COUNT(CASE WHEN agent_performance IN ('ممتاز', 'جيد') THEN 1 END) as good_performance,
        COUNT(CASE WHEN main_status = 'تم السداد' THEN 1 END) as payments_completed,
        COUNT(CASE WHEN main_status = 'العميل وعد بالسداد' THEN 1 END) as payment_promises
    FROM vapi_call_analysiss ca
    JOIN vapi_call_logs cl ON ca.call_id = cl.call_id
    WHERE cl.start_time BETWEEN ? AND ?
");
$analysis_stats->execute([$date_from . ' 00:00:00', $date_to . ' 23:59:59']);
$analysis = $analysis_stats->fetch(PDO::FETCH_ASSOC);

// 3. AGENT PERFORMANCE (Enhanced with more metrics)
$agent_performance = $pdo->prepare("
    SELECT 
        cl.assistant_name,
        COUNT(*) as total_calls,
        COUNT(CASE WHEN cl.success_eval = 'Success' THEN 1 END) as successful_calls,
        AVG(cl.duration_seconds) as avg_duration,
        SUM(cl.cost_usd) as total_cost,
        MIN(cl.start_time) as first_call,
        MAX(cl.start_time) as last_call,
        COUNT(CASE WHEN cl.duration_seconds > 180 THEN 1 END) as quality_calls,
        COUNT(ca.call_id) as analyzed_calls,
        COUNT(CASE WHEN ca.customer_satisfaction IN ('راضي جداً', 'راضي') THEN 1 END) as satisfied_customers,
        COUNT(CASE WHEN ca.agent_performance IN ('ممتاز', 'جيد') THEN 1 END) as good_performance_count
    FROM vapi_call_logs cl
    LEFT JOIN vapi_call_analysiss ca ON cl.call_id = ca.call_id
    WHERE cl.start_time BETWEEN ? AND ?
    GROUP BY cl.assistant_name
    ORDER BY total_calls DESC
");
$agent_performance->execute([$date_from . ' 00:00:00', $date_to . ' 23:59:59']);
$agents = $agent_performance->fetchAll(PDO::FETCH_ASSOC);

// 4. DAILY CALL VOLUME (Fixed for better chart data)
$daily_calls = $pdo->prepare("
    SELECT 
        DATE(start_time) as call_date,
        COUNT(*) as calls_count,
        COUNT(CASE WHEN success_eval = 'Success' THEN 1 END) as successful_count,
        COUNT(CASE WHEN success_eval = 'Fail' THEN 1 END) as failed_count,
        SUM(cost_usd) as daily_cost,
        AVG(duration_seconds) as avg_duration,
        COUNT(CASE WHEN call_type = 'Inbound' THEN 1 END) as inbound_count,
        COUNT(CASE WHEN call_type = 'Outbound' THEN 1 END) as outbound_count
    FROM vapi_call_logs
    WHERE start_time BETWEEN ? AND ?
    GROUP BY DATE(start_time)
    ORDER BY call_date ASC
");
$daily_calls->execute([$date_from . ' 00:00:00', $date_to . ' 23:59:59']);
$daily_data = $daily_calls->fetchAll(PDO::FETCH_ASSOC);

// 5. CALL STATUS BREAKDOWN (Raw Call Logs)
$call_status_breakdown = $pdo->prepare("
    SELECT 
        CASE 
            WHEN success_eval = 'Success' THEN 'ناجح'
            WHEN success_eval = 'Fail' THEN 'فاشل' 
            ELSE 'غير محدد'
        END as call_status,
        COUNT(*) as count
    FROM vapi_call_logs
    WHERE start_time BETWEEN ? AND ?
    GROUP BY success_eval
    ORDER BY count DESC
");
$call_status_breakdown->execute([$date_from . ' 00:00:00', $date_to . ' 23:59:59']);
$call_statuses = $call_status_breakdown->fetchAll(PDO::FETCH_ASSOC);

// 6. CALL DURATION DISTRIBUTION
$duration_distribution = $pdo->prepare("
    SELECT 
        CASE 
            WHEN duration_seconds < 30 THEN 'أقل من 30 ثانية'
            WHEN duration_seconds < 120 THEN '30 ثانية - 2 دقيقة'
            WHEN duration_seconds < 300 THEN '2-5 دقائق'
            WHEN duration_seconds < 600 THEN '5-10 دقائق'
            ELSE 'أكثر من 10 دقائق'
        END as duration_range,
        COUNT(*) as count
    FROM vapi_call_logs
    WHERE start_time BETWEEN ? AND ? AND duration_seconds IS NOT NULL
    GROUP BY 
        CASE 
            WHEN duration_seconds < 30 THEN 'أقل من 30 ثانية'
            WHEN duration_seconds < 120 THEN '30 ثانية - 2 دقيقة'
            WHEN duration_seconds < 300 THEN '2-5 دقائق'
            WHEN duration_seconds < 600 THEN '5-10 دقائق'
            ELSE 'أكثر من 10 دقائق'
        END
    ORDER BY count DESC
");
$duration_distribution->execute([$date_from . ' 00:00:00', $date_to . ' 23:59:59']);
$duration_data = $duration_distribution->fetchAll(PDO::FETCH_ASSOC);

// 7. HOURLY CALL DISTRIBUTION (Fixed to ensure data exists)
$hourly_distribution = $pdo->prepare("
    SELECT 
        HOUR(start_time) as hour_of_day,
        COUNT(*) as calls_count,
        AVG(duration_seconds) as avg_duration
    FROM vapi_call_logs
    WHERE start_time BETWEEN ? AND ?
    GROUP BY HOUR(start_time)
    ORDER BY hour_of_day
");
$hourly_distribution->execute([$date_from . ' 00:00:00', $date_to . ' 23:59:59']);
$hourly_data = $hourly_distribution->fetchAll(PDO::FETCH_ASSOC);

// 8. COST ANALYSIS
$cost_analysis = $pdo->prepare("
    SELECT 
        assistant_name,
        SUM(cost_usd) as total_cost,
        AVG(cost_usd) as avg_cost_per_call,
        COUNT(*) as calls_count,
        SUM(cost_usd) / COUNT(*) as efficiency_ratio
    FROM vapi_call_logs
    WHERE start_time BETWEEN ? AND ? AND cost_usd IS NOT NULL
    GROUP BY assistant_name
    ORDER BY total_cost DESC
");
$cost_analysis->execute([$date_from . ' 00:00:00', $date_to . ' 23:59:59']);
$cost_data = $cost_analysis->fetchAll(PDO::FETCH_ASSOC);

// Calculate percentages and metrics
$success_rate = $call_stats['total_calls'] ? round(($call_stats['successful_calls'] / $call_stats['total_calls']) * 100, 1) : 0;
$analysis_coverage = $call_stats['total_calls'] ? round(($analysis['analyzed_calls'] / $call_stats['total_calls']) * 100, 1) : 0;
$satisfaction_rate = $analysis['analyzed_calls'] ? round(($analysis['satisfied_customers'] / $analysis['analyzed_calls']) * 100, 1) : 0;
$inbound_percentage = $call_stats['total_calls'] ? round(($call_stats['inbound_calls'] / $call_stats['total_calls']) * 100, 1) : 0;
$avg_calls_per_day = count($daily_data) ? round($call_stats['total_calls'] / count($daily_data), 1) : 0;

// Debug: Check if we have data
$has_call_data = $call_stats['total_calls'] > 0;
$has_daily_data = count($daily_data) > 0;
$has_hourly_data = count($hourly_data) > 0;
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="utf-8">
    <title>لوحة تحكم المدير الشاملة - محسنة</title>
    <meta name="viewport" content="width=device-width,initial-scale=1">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <script src="/chart.js"></script>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body { 
            font-family: 'Segoe UI', 'Cairo', Tahoma, Arial, sans-serif; 
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            direction: rtl;
            color: #333;
        }
        
        .header {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(15px);
            padding: 20px 0;
            box-shadow: 0 4px 20px rgba(0,0,0,0.1);
            margin-bottom: 30px;
            position: sticky;
            top: 0;
            z-index: 100;
        }
        
        .container { 
            max-width: 1600px; 
            margin: 0 auto; 
            padding: 0 20px;
        }
        
        .header-content {
            display: flex;
            justify-content: space-between;
            align-items: center;
            flex-wrap: wrap;
            gap: 20px;
        }
        
        h1 {
            color: #2d3748;
            font-size: 2.2rem;
            font-weight: 700;
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .date-filters {
            display: flex;
            gap: 15px;
            align-items: center;
            flex-wrap: wrap;
        }
        
        .date-filters input {
            padding: 10px 15px;
            border: 2px solid #e2e8f0;
            border-radius: 8px;
            background: white;
            font-size: 14px;
        }
        
        .btn {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 10px 20px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-size: 14px;
            font-weight: 600;
            transition: all 0.3s ease;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }
        
        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(0,0,0,0.2);
        }
        
        .main-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(320px, 1fr));
            gap: 25px;
            margin-bottom: 30px;
        }
        
        .card {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(15px);
            border-radius: 20px;
            padding: 30px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
            transition: all 0.3s ease;
            border: 1px solid rgba(255,255,255,0.2);
        }
        
        .card:hover {
            transform: translateY(-5px);
            box-shadow: 0 15px 40px rgba(0,0,0,0.15);
        }
        
        .card h3 {
            color: #2d3748;
            font-size: 1.3rem;
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            gap: 10px;
            border-bottom: 2px solid #e2e8f0;
            padding-bottom: 10px;
        }
        
        .stat-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 20px;
        }
        
        .stat-item {
            text-align: center;
            padding: 20px;
            background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
            border-radius: 12px;
            border-right: 4px solid #667eea;
            transition: all 0.3s ease;
        }
        
        .stat-item:hover {
            transform: scale(1.05);
            box-shadow: 0 8px 25px rgba(0,0,0,0.1);
        }
        
        .stat-number {
            font-size: 2.2rem;
            font-weight: 700;
            color: #2d3748;
            margin-bottom: 5px;
            display: block;
        }
        
        .stat-label {
            color: #718096;
            font-size: 13px;
            font-weight: 500;
        }
        
        .chart-container {
            position: relative;
            height: 300px;
            margin-top: 20px;
        }
        
        .debug-info {
            background: #f0f9ff;
            border: 1px solid #0ea5e9;
            border-radius: 8px;
            padding: 15px;
            margin: 20px 0;
            font-size: 14px;
        }
        
        .debug-info h4 {
            color: #0369a1;
            margin-bottom: 10px;
        }
        
        .progress-bar {
            width: 100%;
            height: 8px;
            background: #e2e8f0;
            border-radius: 4px;
            overflow: hidden;
            margin: 10px 0;
        }
        
        .progress-fill {
            height: 100%;
            background: linear-gradient(90deg, #667eea 0%, #764ba2 100%);
            border-radius: 4px;
            transition: width 0.3s ease;
        }
        
        .navigation-tabs {
            display: flex;
            gap: 10px;
            margin-bottom: 30px;
            background: rgba(255,255,255,0.9);
            padding: 15px;
            border-radius: 15px;
            backdrop-filter: blur(10px);
        }
        
        .tab-btn {
            padding: 12px 24px;
            background: transparent;
            border: 2px solid #e2e8f0;
            border-radius: 10px;
            color: #4a5568;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }
        
        .tab-btn.active {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-color: transparent;
        }
        
        .tab-btn:hover {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-color: transparent;
        }
        
        @media (max-width: 768px) {
            .header-content {
                flex-direction: column;
                align-items: stretch;
            }
            
            .date-filters {
                justify-content: center;
            }
            
            .main-grid {
                grid-template-columns: 1fr;
                gap: 20px;
            }
            
            .stat-grid {
                grid-template-columns: repeat(2, 1fr);
            }
            
            h1 {
                font-size: 1.8rem;
                text-align: center;
            }
            
            .card {
                padding: 20px;
            }
            
            .navigation-tabs {
                flex-direction: column;
            }
        }
    </style>
</head>
<body>
    <div class="header">
        <div class="container">
            <div class="header-content">
                <h1>
                    <i class="fas fa-tachometer-alt"></i>
                    لوحة تحكم المدير الشاملة
                </h1>
                
                <form method="get" class="date-filters">
                    <input type="date" name="date_from" value="<?= h($date_from) ?>">
                    <span>إلى</span>
                    <input type="date" name="date_to" value="<?= h($date_to) ?>">
                    <button type="submit" class="btn">
                        <i class="fas fa-filter"></i> تطبيق
                    </button>
                </form>
            </div>
        </div>
    </div>

    <div class="container">
        <!-- Navigation Tabs -->
        <div class="navigation-tabs">
            <a href="call_logs.php" class="tab-btn">
                <i class="fas fa-list"></i> سجل المكالمات
            </a>
            <a href="call_analyzer.php" class="tab-btn">
                <i class="fas fa-brain"></i> محلل المكالمات
            </a>
            <a href="analysis_dashboard.php" class="tab-btn">
                <i class="fas fa-chart-line"></i> لوحة التحليل
            </a>
            <span class="tab-btn active">
                <i class="fas fa-tachometer-alt"></i> لوحة المدير
            </span>
        </div>

        <!-- Debug Information -->
        <?php if (!$has_call_data): ?>
        <div class="debug-info">
            <h4>🔍 معلومات التشخيص:</h4>
            <p><strong>إجمالي المكالمات:</strong> <?= $call_stats['total_calls'] ?></p>
            <p><strong>فترة البحث:</strong> من <?= $date_from ?> إلى <?= $date_to ?></p>
            <p><strong>البيانات اليومية:</strong> <?= count($daily_data) ?> يوم</p>
            <p><strong>البيانات بالساعة:</strong> <?= count($hourly_data) ?> ساعة</p>
            <?php if ($call_stats['total_calls'] == 0): ?>
            <p style="color: #dc2626;"><strong>⚠️ لا توجد مكالمات في الفترة المحددة - جرب فترة زمنية أوسع</strong></p>
            <?php endif; ?>
        </div>
        <?php endif; ?>

        <!-- Main Statistics Grid -->
        <div class="main-grid">
            <!-- Raw Call Logs Performance -->
            <div class="card">
                <h3><i class="fas fa-chart-bar"></i> إحصائيات المكالمات الأساسية</h3>
                <div class="stat-grid">
                    <div class="stat-item">
                        <span class="stat-number"><?= number_format($call_stats['total_calls']) ?></span>
                        <div class="stat-label">إجمالي المكالمات</div>
                    </div>
                    <div class="stat-item">
                        <span class="stat-number"><?= $success_rate ?>%</span>
                        <div class="stat-label">معدل النجاح</div>
                    </div>
                    <div class="stat-item">
                        <span class="stat-number"><?= formatMoney($call_stats['total_cost']) ?></span>
                        <div class="stat-label">إجمالي التكلفة</div>
                    </div>
                    <div class="stat-item">
                        <span class="stat-number"><?= formatDuration($call_stats['avg_duration']) ?></span>
                        <div class="stat-label">متوسط المدة</div>
                    </div>
                </div>
                
                <div class="progress-bar">
                    <div class="progress-fill" style="width: <?= $success_rate ?>%"></div>
                </div>
                <small>معدل نجاح المكالمات: <?= $success_rate ?>%</small>
            </div>

            <!-- Call Types & Activity -->
            <div class="card">
                <h3><i class="fas fa-phone-alt"></i> تفاصيل نوع المكالمات</h3>
                <div class="stat-grid">
                    <div class="stat-item">
                        <span class="stat-number"><?= number_format($call_stats['inbound_calls']) ?></span>
                        <div class="stat-label">مكالمات واردة</div>
                    </div>
                    <div class="stat-item">
                        <span class="stat-number"><?= number_format($call_stats['outbound_calls']) ?></span>
                        <div class="stat-label">مكالمات صادرة</div>
                    </div>
                    <div class="stat-item">
                        <span class="stat-number"><?= $inbound_percentage ?>%</span>
                        <div class="stat-label">نسبة الواردة</div>
                    </div>
                    <div class="stat-item">
                        <span class="stat-number"><?= $avg_calls_per_day ?></span>
                        <div class="stat-label">متوسط يومي</div>
                    </div>
                </div>
            </div>

            <!-- Call Quality Analysis -->
            <div class="card">
                <h3><i class="fas fa-clock"></i> تحليل جودة المكالمات</h3>
                <div class="stat-grid">
                    <div class="stat-item">
                        <span class="stat-number"><?= number_format($call_stats['very_short_calls']) ?></span>
                        <div class="stat-label">مكالمات قصيرة جداً</div>
                    </div>
                    <div class="stat-item">
                        <span class="stat-number"><?= number_format($call_stats['long_calls']) ?></span>
                        <div class="stat-label">مكالمات طويلة</div>
                    </div>
                    <div class="stat-item">
                        <span class="stat-number"><?= number_format($call_stats['active_agents']) ?></span>
                        <div class="stat-label">وكلاء نشطون</div>
                    </div>
                    <div class="stat-item">
                        <span class="stat-number"><?= number_format($call_stats['unique_customers']) ?></span>
                        <div class="stat-label">عملاء مميزون</div>
                    </div>
                </div>
            </div>

            <!-- AI Analysis Results -->
            <div class="card">
                <h3><i class="fas fa-brain"></i> نتائج التحليل الذكي</h3>
                <div class="stat-grid">
                    <div class="stat-item">
                        <span class="stat-number"><?= number_format($analysis['analyzed_calls']) ?></span>
                        <div class="stat-label">مكالمة محللة</div>
                    </div>
                    <div class="stat-item">
                        <span class="stat-number"><?= $satisfaction_rate ?>%</span>
                        <div class="stat-label">معدل الرضا</div>
                    </div>
                    <div class="stat-item">
                        <span class="stat-number"><?= number_format($analysis['payments_completed']) ?></span>
                        <div class="stat-label">مدفوعات مكتملة</div>
                    </div>
                    <div class="stat-item">
                        <span class="stat-number"><?= number_format($analysis['needs_followup']) ?></span>
                        <div class="stat-label">يحتاج متابعة</div>
                    </div>
                </div>
                
                <div class="progress-bar">
                    <div class="progress-fill" style="width: <?= $analysis_coverage ?>%"></div>
                </div>
                <small>تغطية التحليل: <?= $analysis_coverage ?>%</small>
            </div>
        </div>

        <!-- Charts Row -->
        <div class="main-grid">
            <!-- Daily Call Volume Chart -->
            <div class="card">
                <h3><i class="fas fa-chart-line"></i> حجم المكالمات اليومي</h3>
                <?php if ($has_daily_data): ?>
                <div class="chart-container">
                    <canvas id="dailyCallsChart"></canvas>
                </div>
                <?php else: ?>
                <div class="debug-info">
                    <p>⚠️ لا توجد بيانات يومية لعرضها في الرسم البياني</p>
                </div>
                <?php endif; ?>
            </div>

            <!-- Call Status Distribution -->
            <div class="card">
                <h3><i class="fas fa-pie-chart"></i> توزيع حالة المكالمات</h3>
                <?php if (count($call_statuses) > 0): ?>
                <div class="chart-container">
                    <canvas id="callStatusChart"></canvas>
                </div>
                <?php else: ?>
                <div class="debug-info">
                    <p>⚠️ لا توجد بيانات حالة المكالمات</p>
                </div>
                <?php endif; ?>
            </div>

            <!-- Hourly Distribution -->
            <div class="card">
                <h3><i class="fas fa-clock"></i> توزيع المكالمات بالساعة</h3>
                <?php if ($has_hourly_data): ?>
                <div class="chart-container">
                    <canvas id="hourlyChart"></canvas>
                </div>
                <?php else: ?>
                <div class="debug-info">
                    <p>⚠️ لا توجد بيانات ساعية لعرضها</p>
                </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- Agent Performance & Cost Analysis -->
        <div class="main-grid">
            <!-- Cost Analysis -->
            <div class="card">
                <h3><i class="fas fa-dollar-sign"></i> تحليل التكلفة</h3>
                <?php if (count($cost_data) > 0): ?>
                <div class="chart-container">
                    <canvas id="costChart"></canvas>
                </div>
                <div style="margin-top: 20px;">
                    <h4>تفاصيل التكلفة:</h4>
                    <?php foreach (array_slice($cost_data, 0, 5) as $agent): ?>
                    <div style="display: flex; justify-content: space-between; margin: 10px 0; padding: 10px; background: #f8f9fa; border-radius: 6px;">
                        <strong><?= h($agent['assistant_name']) ?></strong>
                        <span><?= formatMoney($agent['total_cost']) ?> (<?= $agent['calls_count'] ?> مكالمة)</span>
                    </div>
                    <?php endforeach; ?>
                </div>
                <?php else: ?>
                <div class="debug-info">
                    <p>⚠️ لا توجد بيانات تكلفة متاحة</p>
                </div>
                <?php endif; ?>
            </div>

            <!-- Duration Distribution -->
            <div class="card">
                <h3><i class="fas fa-hourglass-half"></i> توزيع مدة المكالمات</h3>
                <?php if (count($duration_data) > 0): ?>
                <div class="chart-container">
                    <canvas id="durationChart"></canvas>
                </div>
                <?php else: ?>
                <div class="debug-info">
                    <p>⚠️ لا توجد بيانات مدة المكالمات</p>
                </div>
                <?php endif; ?>
            </div>

            <!-- Quick Stats Summary -->
            <div class="card">
                <h3><i class="fas fa-info-circle"></i> ملخص سريع</h3>
                <div style="space-y: 15px;">
                    <div class="stat-item" style="margin-bottom: 15px;">
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <span>أول مكالمة:</span>
                            <strong><?= $call_stats['first_call_date'] ? date('M j, Y', strtotime($call_stats['first_call_date'])) : 'N/A' ?></strong>
                        </div>
                    </div>
                    <div class="stat-item" style="margin-bottom: 15px;">
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <span>آخر مكالمة:</span>
                            <strong><?= $call_stats['last_call_date'] ? date('M j, Y H:i', strtotime($call_stats['last_call_date'])) : 'N/A' ?></strong>
                        </div>
                    </div>
                    <div class="stat-item" style="margin-bottom: 15px;">
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <span>متوسط التكلفة/مكالمة:</span>
                            <strong><?= formatMoney($call_stats['avg_cost']) ?></strong>
                        </div>
                    </div>
                    <div class="stat-item">
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <span>إجمالي ساعات المكالمات:</span>
                            <strong><?= formatDuration($call_stats['total_duration']) ?></strong>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Agent Performance Table -->
        <?php if (count($agents) > 0): ?>
        <div class="card">
            <h3><i class="fas fa-users"></i> أداء الوكلاء التفصيلي</h3>
            <div style="overflow-x: auto;">
                <table style="width: 100%; border-collapse: collapse; background: white; border-radius: 12px; overflow: hidden;">
                    <thead>
                        <tr style="background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);">
                            <th style="padding: 15px; text-align: right; border-bottom: 1px solid #e2e8f0;">اسم الوكيل</th>
                            <th style="padding: 15px; text-align: center; border-bottom: 1px solid #e2e8f0;">إجمالي المكالمات</th>
                            <th style="padding: 15px; text-align: center; border-bottom: 1px solid #e2e8f0;">المكالمات الناجحة</th>
                            <th style="padding: 15px; text-align: center; border-bottom: 1px solid #e2e8f0;">معدل النجاح</th>
                            <th style="padding: 15px; text-align: center; border-bottom: 1px solid #e2e8f0;">متوسط المدة</th>
                            <th style="padding: 15px; text-align: center; border-bottom: 1px solid #e2e8f0;">إجمالي التكلفة</th>
                            <th style="padding: 15px; text-align: center; border-bottom: 1px solid #e2e8f0;">مكالمات الجودة</th>
                            <th style="padding: 15px; text-align: center; border-bottom: 1px solid #e2e8f0;">الأداء العام</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($agents as $agent): 
                            $agent_success_rate = $agent['total_calls'] ? round(($agent['successful_calls'] / $agent['total_calls']) * 100, 1) : 0;
                            $quality_rate = $agent['total_calls'] ? round(($agent['quality_calls'] / $agent['total_calls']) * 100, 1) : 0;
                            
                            // Performance badge
                            $performance_class = 'background: #fef5e7; color: #975a16;'; // average
                            if ($agent_success_rate >= 80) $performance_class = 'background: #c6f6d5; color: #22543d;'; // excellent
                            elseif ($agent_success_rate >= 60) $performance_class = 'background: #bee3f8; color: #2a69ac;'; // good
                            elseif ($agent_success_rate < 40) $performance_class = 'background: #fed7d7; color: #742a2a;'; // poor
                        ?>
                        <tr style="border-bottom: 1px solid #f3f4f6;">
                            <td style="padding: 15px; font-weight: 600;"><?= h($agent['assistant_name']) ?></td>
                            <td style="padding: 15px; text-align: center;"><?= number_format($agent['total_calls']) ?></td>
                            <td style="padding: 15px; text-align: center;"><?= number_format($agent['successful_calls']) ?></td>
                            <td style="padding: 15px; text-align: center;">
                                <span style="padding: 4px 10px; border-radius: 12px; font-size: 12px; font-weight: 600; <?= $performance_class ?>">
                                    <?= $agent_success_rate ?>%
                                </span>
                            </td>
                            <td style="padding: 15px; text-align: center;"><?= formatDuration($agent['avg_duration']) ?></td>
                            <td style="padding: 15px; text-align: center;"><?= formatMoney($agent['total_cost']) ?></td>
                            <td style="padding: 15px; text-align: center;"><?= $quality_rate ?>%</td>
                            <td style="padding: 15px; text-align: center;">
                                <?php
                                $overall_score = ($agent_success_rate + $quality_rate) / 2;
                                if ($overall_score >= 75) echo '<span style="padding: 4px 10px; border-radius: 12px; font-size: 12px; font-weight: 600; background: #c6f6d5; color: #22543d;">ممتاز</span>';
                                elseif ($overall_score >= 60) echo '<span style="padding: 4px 10px; border-radius: 12px; font-size: 12px; font-weight: 600; background: #bee3f8; color: #2a69ac;">جيد</span>';
                                elseif ($overall_score >= 40) echo '<span style="padding: 4px 10px; border-radius: 12px; font-size: 12px; font-weight: 600; background: #fef5e7; color: #975a16;">متوسط</span>';
                                else echo '<span style="padding: 4px 10px; border-radius: 12px; font-size: 12px; font-weight: 600; background: #fed7d7; color: #742a2a;">يحتاج تحسين</span>';
                                ?>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
        <?php endif; ?>

        <!-- Quick Actions -->
        <div class="card">
            <h3><i class="fas fa-bolt"></i> إجراءات سريعة</h3>
            <div style="display: flex; gap: 15px; flex-wrap: wrap;">
                <a href="call_analyzer.php" class="btn">
                    <i class="fas fa-play"></i> تشغيل التحليل التلقائي
                </a>
                <a href="call_logs.php" class="btn" style="background: linear-gradient(135deg, #48bb78 0%, #38a169 100%);">
                    <i class="fas fa-list"></i> عرض سجل المكالمات
                </a>
                <a href="analysis_export.php" class="btn" style="background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);">
                    <i class="fas fa-file-excel"></i> تصدير التقارير
                </a>
                <button onclick="location.reload()" class="btn" style="background: linear-gradient(135deg, #6b7280 0%, #4b5563 100%);">
                    <i class="fas fa-sync-alt"></i> تحديث البيانات
                </button>
            </div>
        </div>
    </div>

    <script>
        // Prepare data for charts with fallbacks
        const dailyData = <?= json_encode($daily_data) ?>;
        const callStatuses = <?= json_encode($call_statuses) ?>;
        const hourlyData = <?= json_encode($hourly_data) ?>;
        const durationData = <?= json_encode($duration_data) ?>;
        const costData = <?= json_encode($cost_data) ?>;
        
        console.log('Chart Data Debug:', {
            chartJsLoaded: typeof Chart !== 'undefined',
            dailyData: dailyData.length,
            callStatuses: callStatuses.length,
            hourlyData: hourlyData.length,
            durationData: durationData.length,
            costData: costData.length
        });

        // Wait for DOM to be ready
        // Updated JavaScript for the charts section
        document.addEventListener('DOMContentLoaded', function() {
            console.log('DOM loaded, initializing charts...');
        
            // 1. Daily Calls Chart - with American date format
            if (dailyData.length > 0) {
                const dailyCtx = document.getElementById('dailyCallsChart');
                if (dailyCtx) {
                    console.log('Creating daily calls chart...');
                    new Chart(dailyCtx.getContext('2d'), {
                        type: 'line',
                        data: {
                            labels: dailyData.map(d => {
                                const date = new Date(d.call_date);
                                // Use American format: MM/DD
                                return date.toLocaleDateString('en-US', { 
                                    month: 'short', 
                                    day: 'numeric' 
                                });
                            }),
                            datasets: [{
                                label: 'إجمالي المكالمات',
                                data: dailyData.map(d => d.calls_count),
                                borderColor: 'rgb(102, 126, 234)',
                                backgroundColor: 'rgba(102, 126, 234, 0.1)',
                                tension: 0.4,
                                fill: true
                            }, {
                                label: 'المكالمات الناجحة',
                                data: dailyData.map(d => d.successful_count),
                                borderColor: 'rgb(16, 185, 129)',
                                backgroundColor: 'rgba(16, 185, 129, 0.1)',
                                tension: 0.4,
                                fill: true
                            }, {
                                label: 'المكالمات الفاشلة',
                                data: dailyData.map(d => d.failed_count),
                                borderColor: 'rgb(239, 68, 68)',
                                backgroundColor: 'rgba(239, 68, 68, 0.1)',
                                tension: 0.4,
                                fill: true
                            }]
                        },
                        options: {
                            responsive: true,
                            maintainAspectRatio: false,
                            plugins: {
                                legend: {
                                    labels: {
                                        usePointStyle: true,
                                        padding: 20
                                    }
                                }
                            },
                            scales: {
                                y: {
                                    beginAtZero: true,
                                    grid: {
                                        color: 'rgba(0,0,0,0.1)'
                                    }
                                },
                                x: {
                                    grid: {
                                        color: 'rgba(0,0,0,0.1)'
                                    }
                                }
                            }
                        }
                    });
                    console.log('✅ Daily calls chart created');
                }
            }
        
            // 2. Call Status Chart - with percentages
            if (callStatuses.length > 0) {
                const statusCtx = document.getElementById('callStatusChart');
                if (statusCtx) {
                    console.log('Creating call status chart...');
                    
                    // Calculate total calls and percentages
                    const totalCalls = callStatuses.reduce((sum, status) => sum + parseInt(status.count), 0);
                    const statusPercentages = callStatuses.map(status => ({
                        ...status,
                        percentage: ((parseInt(status.count) / totalCalls) * 100).toFixed(1)
                    }));
                    
                    new Chart(statusCtx.getContext('2d'), {
                        type: 'doughnut',
                        data: {
                            labels: statusPercentages.map(d => `${d.call_status} (${d.percentage}%)`),
                            datasets: [{
                                data: statusPercentages.map(d => parseFloat(d.percentage)),
                                backgroundColor: [
                                    '#10b981', // ناجح
                                    '#ef4444', // فاشل
                                    '#f59e0b'  // غير محدد
                                ],
                                borderWidth: 0
                            }]
                        },
                        options: {
                            responsive: true,
                            maintainAspectRatio: false,
                            plugins: {
                                legend: {
                                    position: 'bottom',
                                    labels: {
                                        usePointStyle: true,
                                        padding: 15
                                    }
                                },
                                tooltip: {
                                    callbacks: {
                                        label: function(context) {
                                            const label = context.label || '';
                                            const value = context.parsed;
                                            const count = callStatuses[context.dataIndex].count;
                                            return `${label}: ${count} مكالمة (${value}%)`;
                                        }
                                    }
                                }
                            }
                        }
                    });
                    console.log('✅ Call status chart with percentages created');
                }
            }
        
            // 3. Hourly Distribution Chart - unchanged
            if (hourlyData.length > 0) {
                const hourlyCtx = document.getElementById('hourlyChart');
                if (hourlyCtx) {
                    console.log('Creating hourly distribution chart...');
                    const completeHourlyData = [];
                    for (let i = 0; i < 24; i++) {
                        const found = hourlyData.find(d => d.hour_of_day == i);
                        completeHourlyData.push({
                            hour_of_day: i,
                            calls_count: found ? found.calls_count : 0
                        });
                    }
                    
                    new Chart(hourlyCtx.getContext('2d'), {
                        type: 'bar',
                        data: {
                            labels: completeHourlyData.map(d => d.hour_of_day + ':00'),
                            datasets: [{
                                label: 'عدد المكالمات',
                                data: completeHourlyData.map(d => d.calls_count),
                                backgroundColor: 'rgba(102, 126, 234, 0.8)',
                                borderColor: 'rgb(102, 126, 234)',
                                borderWidth: 1
                            }]
                        },
                        options: {
                            responsive: true,
                            maintainAspectRatio: false,
                            plugins: {
                                legend: {
                                    display: false
                                }
                            },
                            scales: {
                                y: {
                                    beginAtZero: true,
                                    grid: {
                                        color: 'rgba(0,0,0,0.1)'
                                    }
                                },
                                x: {
                                    grid: {
                                        color: 'rgba(0,0,0,0.1)'
                                    }
                                }
                            }
                        }
                    });
                    console.log('✅ Hourly distribution chart created');
                }
            }
        
            // 4. Duration Distribution Chart - unchanged
            if (durationData.length > 0) {
                const durationCtx = document.getElementById('durationChart');
                if (durationCtx) {
                    console.log('Creating duration distribution chart...');
                    new Chart(durationCtx.getContext('2d'), {
                        type: 'pie',
                        data: {
                            labels: durationData.map(d => d.duration_range),
                            datasets: [{
                                data: durationData.map(d => d.count),
                                backgroundColor: [
                                    '#ef4444', // أقل من 30 ثانية
                                    '#f59e0b', // 30 ثانية - 2 دقيقة
                                    '#10b981', // 2-5 دقائق
                                    '#3b82f6', // 5-10 دقائق
                                    '#8b5cf6'  // أكثر من 10 دقائق
                                ],
                                borderWidth: 2,
                                borderColor: '#fff'
                            }]
                        },
                        options: {
                            responsive: true,
                            maintainAspectRatio: false,
                            plugins: {
                                legend: {
                                    position: 'bottom',
                                    labels: {
                                        usePointStyle: true,
                                        padding: 15
                                    }
                                }
                            }
                        }
                    });
                    console.log('✅ Duration distribution chart created');
                }
            }
        
            // 5. Cost Analysis Chart - unchanged
            if (costData.length > 0) {
                const costCtx = document.getElementById('costChart');
                if (costCtx) {
                    console.log('Creating cost analysis chart...');
                    new Chart(costCtx.getContext('2d'), {
                        type: 'bar',
                        data: {
                            labels: costData.slice(0, 8).map(d => d.assistant_name),
                            datasets: [{
                                label: 'إجمالي التكلفة ($)',
                                data: costData.slice(0, 8).map(d => parseFloat(d.total_cost)),
                                backgroundColor: 'rgba(59, 130, 246, 0.8)',
                                borderColor: 'rgb(59, 130, 246)',
                                borderWidth: 1
                            }]
                        },
                        options: {
                            responsive: true,
                            maintainAspectRatio: false,
                            plugins: {
                                legend: {
                                    display: false
                                }
                            },
                            scales: {
                                y: {
                                    beginAtZero: true,
                                    grid: {
                                        color: 'rgba(0,0,0,0.1)'
                                    },
                                    ticks: {
                                        callback: function(value) {
                                            return '$' + value.toFixed(2);
                                        }
                                    }
                                },
                                x: {
                                    grid: {
                                        color: 'rgba(0,0,0,0.1)'
                                    }
                                }
                            }
                        }
                    });
                    console.log('✅ Cost analysis chart created');
                }
            }
        
            console.log('🎉 All charts initialization completed!');
        });
        
        // Auto-refresh dashboard every 10 minutes
        setInterval(function() {
            console.log('Auto-refreshing dashboard...');
            location.reload();
        }, 600000);
        
        // Add loading states to buttons
        document.querySelectorAll('.btn').forEach(btn => {
            btn.addEventListener('click', function(e) {
                if (this.href && this.href.includes('analyzer')) {
                    this.innerHTML = '<i class="fas fa-spinner fa-spin"></i> جاري التحميل...';
                }
            });
        });
    </script>
</body>
</html>
