<?php
// Show call logs from MySQL with filters, pagination, and XLSX export.
// Cost column has been removed from display and export
require_once __DIR__ . '/vapi_config.php';

// Start session for authentication
session_start();

// --- Optional: set your local timezone for display (KSA) ---
date_default_timezone_set('Asia/Riyadh');

// User credentials - stored securely with hashed passwords
$users = [
    'admin' => password_hash('CallLog2024!', PASSWORD_DEFAULT),
    'manager' => password_hash('Manager@456', PASSWORD_DEFAULT),
    'viewer' => password_hash('Viewer#789', PASSWORD_DEFAULT)
];

// Handle login
if (isset($_POST['login'])) {
    $username = trim($_POST['username'] ?? '');
    $password = $_POST['password'] ?? '';
    
    if (isset($users[$username]) && password_verify($password, $users[$username])) {
        $_SESSION['logged_in'] = true;
        $_SESSION['username'] = $username;
        $_SESSION['login_time'] = time();
        header('Location: ' . $_SERVER['PHP_SELF']);
        exit;
    } else {
        $login_error = 'Invalid username or password';
    }
}

// Handle logout
if (isset($_GET['logout'])) {
    session_destroy();
    header('Location: ' . $_SERVER['PHP_SELF']);
    exit;
}

// Check if user is logged in
if (!isset($_SESSION['logged_in'])) {
    // Show login form
    ?>
    <!DOCTYPE html>
    <html lang="en">
    <head>
        <meta charset="utf-8">
        <title>Call Logs - Secure Login</title>
        <meta name="viewport" content="width=device-width,initial-scale=1">
        <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
        <style>
            * {
                margin: 0;
                padding: 0;
                box-sizing: border-box;
            }
            
            body { 
                font-family: 'Segoe UI', Tahoma, Arial, sans-serif; 
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                min-height: 100vh;
                display: flex;
                align-items: center;
                justify-content: center;
                color: #333;
            }
            
            .login-container {
                background: rgba(255, 255, 255, 0.95);
                backdrop-filter: blur(15px);
                border-radius: 24px;
                padding: 50px 40px;
                box-shadow: 0 20px 60px rgba(0,0,0,0.15);
                width: 100%;
                max-width: 420px;
                text-align: center;
                animation: slideUp 0.6s ease-out;
            }
            
            @keyframes slideUp {
                from {
                    opacity: 0;
                    transform: translateY(30px);
                }
            }
            
            .login-header {
                margin-bottom: 35px;
            }
            
            .login-header i {
                font-size: 3.5rem;
                color: #667eea;
                margin-bottom: 15px;
            }
            
            .login-header h1 {
                font-size: 2rem;
                color: #2d3748;
                margin-bottom: 8px;
                font-weight: 700;
            }
            
            .login-header p {
                color: #718096;
                font-size: 0.95rem;
            }
            
            .error-message {
                background: #fee;
                color: #c53030;
                padding: 14px 18px;
                border-radius: 12px;
                margin-bottom: 25px;
                display: flex;
                align-items: center;
                gap: 10px;
                font-size: 0.9rem;
                border: 1px solid #fc8181;
                animation: shake 0.5s;
            }
            
            .form-group {
                margin-bottom: 22px;
                text-align: left;
            }
            
            .form-group label {
                display: block;
                margin-bottom: 8px;
                color: #4a5568;
                font-weight: 600;
                font-size: 0.92rem;
            }
            
            .form-group label i {
                margin-right: 6px;
                color: #667eea;
            }
            
            .form-group input {
                width: 100%;
                padding: 14px 16px;
                border: 2px solid #e2e8f0;
                border-radius: 12px;
                font-size: 0.95rem;
                transition: all 0.3s ease;
                background: #f7fafc;
            }
            
            .form-group input:focus {
                outline: none;
                border-color: #667eea;
                background: white;
                box-shadow: 0 0 0 4px rgba(102,126,234,0.1);
            }
            
            .login-btn {
                width: 100%;
                padding: 15px;
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                color: white;
                border: none;
                border-radius: 12px;
                font-size: 1rem;
                font-weight: 600;
                cursor: pointer;
                transition: all 0.3s ease;
                margin-top: 10px;
            }
            
            .login-btn:hover {
                transform: translateY(-2px);
                box-shadow: 0 8px 25px rgba(102,126,234,0.4);
            }
            
            .login-btn:active {
                transform: translateY(0);
            }
            
            .login-btn i {
                margin-right: 8px;
            }
            
            @keyframes shake {
                0%, 100% { transform: translateX(0); }
                25% { transform: translateX(-5px); }
                75% { transform: translateX(5px); }
            }
            
            .footer {
                margin-top: 30px;
                color: #a0aec0;
                font-size: 13px;
            }
            
            @media (max-width: 480px) {
                .login-container {
                    margin: 20px;
                    padding: 40px 30px;
                }
                
                .login-header h1 {
                    font-size: 1.6rem;
                }
            }
        </style>
    </head>
    <body>
        <div class="login-container">
            <div class="login-header">
                <i class="fas fa-shield-alt"></i>
                <h1>Call Logs Dashboard</h1>
                <p>Enter your credentials to continue</p>
            </div>
            
            <?php if (isset($login_error)): ?>
            <div class="error-message">
                <i class="fas fa-exclamation-triangle"></i> <?= htmlspecialchars($login_error) ?>
            </div>
            <?php endif; ?>
            
            <form method="POST">
                <div class="form-group">
                    <label for="username">
                        <i class="fas fa-user"></i> Username
                    </label>
                    <input type="text" id="username" name="username" required 
                           placeholder="Enter your username" value="<?= htmlspecialchars($_POST['username'] ?? '') ?>">
                </div>
                
                <div class="form-group">
                    <label for="password">
                        <i class="fas fa-lock"></i> Password
                    </label>
                    <input type="password" id="password" name="password" required 
                           placeholder="Enter your password">
                </div>
                
                <button type="submit" name="login" class="login-btn">
                    <i class="fas fa-sign-in-alt"></i> Sign In
                </button>
            </form>
            
            <div class="footer">
                <i class="fas fa-lock"></i> Secure authentication required
            </div>
        </div>
    </body>
    </html>
    <?php
    exit;
}

// --- helpers ---
function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
function fmt_duration($s){
  if ($s === null || $s === '') return '';
  $s = (int)$s; $m = intdiv($s, 60); $r = $s % 60;
  if ($m > 0) return $m . 'm ' . $r . 's';
  return $r . 's';
}

// --- inputs ---
$page     = max(1, (int)($_GET['page'] ?? 1));
$perPage  = (int)($_GET['per_page'] ?? 50);
if (!in_array($perPage, [25,50,100,200], true)) $perPage = 50;
$q        = trim((string)($_GET['q'] ?? ''));
$from     = trim((string)($_GET['date_from'] ?? '')); // YYYY-MM-DD
$to       = trim((string)($_GET['date_to']   ?? '')); // YYYY-MM-DD
$success_filter = trim((string)($_GET['success_filter'] ?? ''));
$call_type_filter = trim((string)($_GET['call_type_filter'] ?? ''));
$export   = isset($_GET['export']) && $_GET['export'] === 'xlsx';

$where = [];
$params = [];

// text search across a few columns
if ($q !== '') {
  $where[] =
    "(call_id LIKE :q OR assistant_name LIKE :q OR assistant_phone LIKE :q
      OR customer_phone LIKE :q OR ended_reason LIKE :q OR success_eval LIKE :q)";
  $params[':q'] = '%' . $q . '%';
}

// date filters (on start_time)
if ($from !== '' && preg_match('/^\d{4}-\d{2}-\d{2}$/', $from)) {
  $where[] = "start_time >= :from";
  $params[':from'] = $from . " 00:00:00";
}
if ($to !== '' && preg_match('/^\d{4}-\d{2}-\d{2}$/', $to)) {
  $where[] = "start_time <= :to";
  $params[':to'] = $to . " 23:59:59";
}

// Success evaluation filter
if ($success_filter !== '') {
  $where[] = "success_eval = :success_filter";
  $params[':success_filter'] = $success_filter;
}

// Call type filter
if ($call_type_filter !== '') {
  $where[] = "call_type = :call_type_filter";
  $params[':call_type_filter'] = $call_type_filter;
}

$whereSql = $where ? ('WHERE ' . implode(' AND ', $where)) : '';

// --- DB ---
try {
    $pdo = new PDO(DB_DSN, DB_USER, DB_PASS, [
      PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
      PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4"
    ]);
} catch (Exception $e) {
    die("Database connection failed. Please check your configuration.");
}

// Get filter options safely
try {
    $success_options = $pdo->query("SELECT DISTINCT success_eval FROM vapi_call_logs WHERE success_eval IS NOT NULL AND success_eval != '' ORDER BY success_eval")->fetchAll(PDO::FETCH_COLUMN);
    $call_type_options = $pdo->query("SELECT DISTINCT call_type FROM vapi_call_logs WHERE call_type IS NOT NULL AND call_type != '' ORDER BY call_type")->fetchAll(PDO::FETCH_COLUMN);
} catch (Exception $e) {
    $success_options = [];
    $call_type_options = [];
}

// total count
$sqlCount = "SELECT COUNT(*) FROM vapi_call_logs $whereSql";
$stmt = $pdo->prepare($sqlCount);
$stmt->execute($params);
$total = (int)$stmt->fetchColumn();

$offset = ($page - 1) * $perPage;

// rows for current page - REMOVED cost_usd from SELECT
$sqlRows = "
  SELECT call_id, assistant_name, assistant_phone, customer_phone, call_type,
         ended_reason, success_eval, start_time, duration_seconds
  FROM vapi_call_logs
  $whereSql
  ORDER BY start_time DESC
  LIMIT :limit OFFSET :offset
";
$stmt = $pdo->prepare($sqlRows);
foreach ($params as $k=>$v) $stmt->bindValue($k, $v);
$stmt->bindValue(':limit',  $perPage, PDO::PARAM_INT);
$stmt->bindValue(':offset', $offset,  PDO::PARAM_INT);
$stmt->execute();
$rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

// XLSX export - Basic Excel format with just data - REMOVED cost_usd
if ($export) {
  // Get ALL rows matching the filters (remove pagination)
  $sqlExport = "
    SELECT call_id, assistant_name, assistant_phone, customer_phone, call_type,
           ended_reason, success_eval, start_time, duration_seconds
    FROM vapi_call_logs
    $whereSql
    ORDER BY start_time DESC
  ";
  $stmtExport = $pdo->prepare($sqlExport);
  $stmtExport->execute($params);
  $exportRows = $stmtExport->fetchAll(PDO::FETCH_ASSOC);
  
  $filename = 'call_logs_' . date('Y-m-d_H-i-s') . '.xlsx';
  
  // Create basic XLSX
  $xlsx_data = createBasicXLSX($exportRows);
  
  header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
  header('Content-Disposition: attachment; filename=' . $filename);
  header('Content-Length: ' . strlen($xlsx_data));
  
  echo $xlsx_data;
  exit;
}

// Function to create basic XLSX file - REMOVED cost column
function createBasicXLSX($rows) {
  // Create temporary directory
  $temp_dir = sys_get_temp_dir() . '/xlsx_' . uniqid();
  mkdir($temp_dir);
  mkdir($temp_dir . '/xl');
  mkdir($temp_dir . '/xl/worksheets');
  mkdir($temp_dir . '/_rels');
  mkdir($temp_dir . '/xl/_rels');
  
  // Create [Content_Types].xml
  file_put_contents($temp_dir . '/[Content_Types].xml', 
    '<?xml version="1.0" encoding="UTF-8" standalone="yes"?>' .
    '<Types xmlns="http://schemas.openxmlformats.org/package/2006/content-types">' .
    '<Default Extension="rels" ContentType="application/vnd.openxmlformats-package.relationships+xml"/>' .
    '<Default Extension="xml" ContentType="application/xml"/>' .
    '<Override PartName="/xl/workbook.xml" ContentType="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet.main+xml"/>' .
    '<Override PartName="/xl/worksheets/sheet1.xml" ContentType="application/vnd.openxmlformats-officedocument.spreadsheetml.worksheet+xml"/>' .
    '</Types>'
  );
  
  // Create _rels/.rels
  file_put_contents($temp_dir . '/_rels/.rels',
    '<?xml version="1.0" encoding="UTF-8" standalone="yes"?>' .
    '<Relationships xmlns="http://schemas.openxmlformats.org/package/2006/relationships">' .
    '<Relationship Id="rId1" Type="http://schemas.openxmlformats.org/officeDocument/2006/relationships/officeDocument" Target="xl/workbook.xml"/>' .
    '</Relationships>'
  );
  
  // Create xl/_rels/workbook.xml.rels
  file_put_contents($temp_dir . '/xl/_rels/workbook.xml.rels',
    '<?xml version="1.0" encoding="UTF-8" standalone="yes"?>' .
    '<Relationships xmlns="http://schemas.openxmlformats.org/package/2006/relationships">' .
    '<Relationship Id="rId1" Type="http://schemas.openxmlformats.org/officeDocument/2006/relationships/worksheet" Target="worksheets/sheet1.xml"/>' .
    '</Relationships>'
  );
  
  // Create xl/workbook.xml
  file_put_contents($temp_dir . '/xl/workbook.xml',
    '<?xml version="1.0" encoding="UTF-8" standalone="yes"?>' .
    '<workbook xmlns="http://schemas.openxmlformats.org/spreadsheetml/2006/main" xmlns:r="http://schemas.openxmlformats.org/officeDocument/2006/relationships">' .
    '<sheets><sheet name="Call Logs" sheetId="1" r:id="rId1"/></sheets>' .
    '</workbook>'
  );
  
  // Create basic worksheet XML - REMOVED Cost USD column
  $worksheet_xml = '<?xml version="1.0" encoding="UTF-8" standalone="yes"?>' .
    '<worksheet xmlns="http://schemas.openxmlformats.org/spreadsheetml/2006/main">' .
    '<sheetData>';
  
  // Add header row - REMOVED Cost USD
  $worksheet_xml .= '<row r="1">' .
    '<c r="A1" t="inlineStr"><is><t>Call ID</t></is></c>' .
    '<c r="B1" t="inlineStr"><is><t>Assistant</t></is></c>' .
    '<c r="C1" t="inlineStr"><is><t>Assistant Phone</t></is></c>' .
    '<c r="D1" t="inlineStr"><is><t>Customer Phone</t></is></c>' .
    '<c r="E1" t="inlineStr"><is><t>Type</t></is></c>' .
    '<c r="F1" t="inlineStr"><is><t>End Reason</t></is></c>' .
    '<c r="G1" t="inlineStr"><is><t>Success</t></is></c>' .
    '<c r="H1" t="inlineStr"><is><t>Start Time</t></is></c>' .
    '<c r="I1" t="inlineStr"><is><t>Duration</t></is></c>' .
    '</row>';
  
  // Add data rows - REMOVED cost column
  $row_num = 2;
  foreach ($rows as $r) {
    $cols = ['A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I'];
    
    $data = [
      htmlspecialchars($r['call_id']),
      htmlspecialchars($r['assistant_name'] ?: 'Unknown'),
      htmlspecialchars($r['assistant_phone'] ?: 'N/A'),
      htmlspecialchars($r['customer_phone'] ?: 'N/A'),
      htmlspecialchars($r['call_type'] ?: 'Unknown'),
      htmlspecialchars($r['ended_reason'] ?: 'N/A'),
      htmlspecialchars($r['success_eval'] ?: 'Unknown'),
      htmlspecialchars($r['start_time'] ?: 'N/A'),
      fmt_duration($r['duration_seconds']) ?: 'N/A'
    ];
    
    $worksheet_xml .= "<row r=\"$row_num\">";
    for ($i = 0; $i < count($data); $i++) {
      $cell_ref = $cols[$i] . $row_num;
      $worksheet_xml .= "<c r=\"$cell_ref\" t=\"inlineStr\"><is><t>{$data[$i]}</t></is></c>";
    }
    $worksheet_xml .= "</row>";
    $row_num++;
  }
  
  $worksheet_xml .= '</sheetData></worksheet>';
  
  file_put_contents($temp_dir . '/xl/worksheets/sheet1.xml', $worksheet_xml);
  
  // Create ZIP file
  $zip = new ZipArchive();
  $zip_filename = $temp_dir . '/output.xlsx';
  $zip->open($zip_filename, ZipArchive::CREATE);
  
  // Add files to ZIP
  $files = new RecursiveIteratorIterator(
    new RecursiveDirectoryIterator($temp_dir),
    RecursiveIteratorIterator::LEAVES_ONLY
  );
  
  foreach ($files as $name => $file) {
    if (!$file->isDir() && $file->getFilename() != 'output.xlsx') {
      $filePath = $file->getRealPath();
      $relativePath = substr($filePath, strlen($temp_dir) + 1);
      $zip->addFile($filePath, $relativePath);
    }
  }
  
  $zip->close();
  
  // Read the ZIP file
  $xlsx_content = file_get_contents($zip_filename);
  
  // Clean up temporary files
  function deleteDir($dirPath) {
    if (!is_dir($dirPath)) return false;
    $files = new RecursiveIteratorIterator(
      new RecursiveDirectoryIterator($dirPath, RecursiveDirectoryIterator::SKIP_DOTS),
      RecursiveIteratorIterator::CHILD_FIRST
    );
    foreach ($files as $fileinfo) {
      $todo = ($fileinfo->isDir() ? 'rmdir' : 'unlink');
      $todo($fileinfo->getRealPath());
    }
    return rmdir($dirPath);
  }
  
  deleteDir($temp_dir);
  
  return $xlsx_content;
}

// Track active filters
$active_filters = [];
if ($q !== '') $active_filters[] = "Search: \"$q\"";
if ($from !== '') $active_filters[] = "From: $from";
if ($to !== '') $active_filters[] = "To: $to";
if ($success_filter !== '') $active_filters[] = "Status: $success_filter";
if ($call_type_filter !== '') $active_filters[] = "Type: $call_type_filter";
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <title>Call Logs Dashboard</title>
    <meta name="viewport" content="width=device-width,initial-scale=1">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body { 
            font-family: 'Segoe UI', Tahoma, Arial, sans-serif; 
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
            min-height: 100vh;
            color: #2d3748;
            padding-bottom: 40px;
        }
        
        .header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 25px 0;
            box-shadow: 0 4px 20px rgba(0,0,0,0.15);
            position: sticky;
            top: 0;
            z-index: 100;
        }
        
        .header-content {
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .header-left h1 {
            font-size: 2.2rem;
            font-weight: 700;
            margin-bottom: 5px;
            display: flex;
            align-items: center;
            gap: 12px;
        }
        
        .subtitle {
            font-size: 1rem;
            opacity: 0.95;
            font-weight: 400;
        }
        
        .user-info {
            display: flex;
            align-items: center;
            gap: 20px;
        }
        
        .welcome-text {
            font-size: 1rem;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .welcome-text i {
            font-size: 1.3rem;
        }
        
        .logout-btn {
            background: rgba(255,255,255,0.2);
            color: white;
            padding: 10px 20px;
            border-radius: 8px;
            text-decoration: none;
            font-weight: 600;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 8px;
            border: 2px solid rgba(255,255,255,0.3);
        }
        
        .logout-btn:hover {
            background: rgba(255,255,255,0.3);
            transform: translateY(-2px);
        }
        
        .container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 0 25px;
        }
        
        .card {
            background: white;
            border-radius: 16px;
            padding: 30px;
            margin: 25px 0;
            box-shadow: 0 4px 20px rgba(0,0,0,0.08);
        }
        
        .card h2 {
            font-size: 1.5rem;
            margin-bottom: 20px;
            color: #2d3748;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .toolbar {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 18px;
            margin-bottom: 25px;
        }
        
        .filter-group {
            display: flex;
            flex-direction: column;
        }
        
        .filter-group label {
            margin-bottom: 6px;
            font-weight: 600;
            color: #4a5568;
            font-size: 0.9rem;
        }
        
        .filter-group input,
        .filter-group select {
            padding: 11px 14px;
            border: 2px solid #e2e8f0;
            border-radius: 10px;
            font-size: 0.95rem;
            transition: all 0.3s ease;
            background: #f7fafc;
        }
        
        .filter-group input:focus,
        .filter-group select:focus {
            outline: none;
            border-color: #667eea;
            background: white;
            box-shadow: 0 0 0 3px rgba(102,126,234,0.1);
        }
        
        .filter-actions {
            grid-column: 1 / -1;
            display: flex;
            gap: 12px;
            flex-wrap: wrap;
        }
        
        .btn {
            padding: 12px 24px;
            border: none;
            border-radius: 10px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: inline-flex;
            align-items: center;
            gap: 8px;
            text-decoration: none;
            font-size: 0.95rem;
        }
        
        .btn-primary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }
        
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(102,126,234,0.4);
        }
        
        .btn-secondary {
            background: #edf2f7;
            color: #4a5568;
        }
        
        .btn-secondary:hover {
            background: #e2e8f0;
            transform: translateY(-2px);
        }
        
        .btn-success {
            background: linear-gradient(135deg, #48bb78 0%, #38a169 100%);
            color: white;
        }
        
        .btn-success:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(72,187,120,0.4);
        }
        
        .filter-summary {
            background: linear-gradient(135deg, #fef5e7 0%, #fdeaa7 100%);
            padding: 15px 20px;
            border-radius: 12px;
            margin-bottom: 20px;
            border-left: 4px solid #f39c12;
            display: flex;
            flex-wrap: wrap;
            align-items: center;
            gap: 12px;
        }
        
        .filter-tag {
            background: white;
            padding: 5px 12px;
            border-radius: 20px;
            font-size: 0.85rem;
            font-weight: 600;
            color: #2d3748;
            box-shadow: 0 2px 8px rgba(0,0,0,0.08);
        }
        
        .stats-bar {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 18px;
            background: linear-gradient(135deg, #e6f2ff 0%, #c3e0ff 100%);
            border-radius: 12px;
            margin-bottom: 20px;
            border-left: 4px solid #3182ce;
        }
        
        .stats-info {
            font-size: 1rem;
            color: #2c5282;
            font-weight: 500;
        }
        
        .stats-highlight {
            font-weight: 700;
            color: #2b6cb0;
            font-size: 1.1rem;
        }
        
        .table-container {
            overflow-x: auto;
            border-radius: 12px;
            box-shadow: 0 2px 12px rgba(0,0,0,0.06);
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
            background: white;
        }
        
        th {
            background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
            padding: 16px 14px;
            text-align: left;
            font-weight: 700;
            color: #2d3748;
            border-bottom: 2px solid #e2e8f0;
            white-space: nowrap;
            font-size: 0.9rem;
        }
        
        th i {
            margin-right: 6px;
            color: #667eea;
        }
        
        td {
            padding: 16px 14px;
            border-bottom: 1px solid #f1f5f9;
            font-size: 0.92rem;
        }
        
        tr:hover {
            background: #f8fafc;
        }
        
        tr:last-child td {
            border-bottom: none;
        }
        
        .call-id {
            color: #667eea;
            text-decoration: none;
            font-weight: 600;
            transition: all 0.2s;
        }
        
        .call-id:hover {
            color: #764ba2;
            text-decoration: underline;
        }
        
        .pill {
            padding: 5px 12px;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 600;
            display: inline-block;
            white-space: nowrap;
        }
        
        .pill.success {
            background: #d4edda;
            color: #155724;
        }
        
        .pill.fail {
            background: #f8d7da;
            color: #721c24;
        }
        
        .pill.inbound {
            background: #cfe2ff;
            color: #084298;
        }
        
        .pill.outbound {
            background: #e7d4ff;
            color: #4a148c;
        }
        
        .duration {
            font-weight: 600;
            color: #4a5568;
        }
        
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: #a0aec0;
        }
        
        .empty-state i {
            font-size: 4rem;
            margin-bottom: 20px;
            opacity: 0.5;
        }
        
        .empty-state h3 {
            font-size: 1.5rem;
            margin-bottom: 10px;
            color: #718096;
        }
        
        .pagination {
            display: flex;
            justify-content: center;
            align-items: center;
            gap: 15px;
            margin-top: 30px;
            padding: 20px;
        }
        
        .pagination a,
        .pagination .current {
            padding: 12px 20px;
            border-radius: 10px;
            text-decoration: none;
            font-weight: 600;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .pagination a {
            background: #edf2f7;
            color: #4a5568;
        }
        
        .pagination a:hover {
            background: #667eea;
            color: white;
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(102,126,234,0.3);
        }
        
        .pagination .current {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            cursor: default;
        }
        
        @media (max-width: 1024px) {
            .header-left h1 {
                font-size: 1.8rem;
            }
            
            .toolbar {
                grid-template-columns: repeat(auto-fit, minmax(180px, 1fr));
            }
            
            .card {
                padding: 20px;
                margin: 15px 0;
            }
            
            .table-container {
                font-size: 12px;
            }
            
            th, td {
                padding: 12px 8px;
            }
        }
        
        @media (max-width: 768px) {
            .header-content {
                flex-direction: column;
                gap: 15px;
                align-items: center;
                text-align: center;
            }
            
            .user-info {
                flex-direction: column;
                gap: 10px;
            }
            
            .toolbar {
                grid-template-columns: 1fr;
                gap: 15px;
            }
            
            .filter-actions {
                flex-direction: column;
                align-items: stretch;
                gap: 10px;
            }
            
            .filter-actions .btn {
                width: 100%;
                justify-content: center;
            }
            
            .filter-summary {
                flex-direction: column;
                align-items: flex-start;
                gap: 10px;
            }
            
            .filter-tag {
                font-size: 11px;
                padding: 3px 8px;
            }
            
            .stats-bar {
                flex-direction: column;
                gap: 10px;
                text-align: center;
            }
            
            .header-left h1 {
                font-size: 1.8rem;
            }
            
            .card {
                padding: 20px;
                margin: 15px 0;
            }
        }
    </style>
</head>
<body>
    <div class="header">
        <div class="container">
            <div class="header-content">
                <div class="header-left">
                    <h1><i class="fas fa-phone-alt"></i> Call Logs Dashboard</h1>
                    <p class="subtitle">Monitor and analyze all your call activities</p>
                </div>
                <div class="user-info">
                    <span class="welcome-text">
                        <i class="fas fa-user-circle"></i> Welcome, <strong><?= htmlspecialchars($_SESSION['username']) ?></strong>
                    </span>
                    <a href="?logout=1" class="logout-btn" onclick="return confirm('Are you sure you want to logout?')">
                        <i class="fas fa-sign-out-alt"></i> Logout
                    </a>
                </div>
            </div>
        </div>
    </div>

    <div class="container">
        <div class="card">
            <h2><i class="fas fa-filter"></i> Advanced Filters</h2>
            <form class="toolbar" method="get">
                <div class="filter-group">
                    <label>Search</label>
                    <input type="text" name="q" placeholder="Search calls..." value="<?=h($q)?>">
                </div>
                
                <div class="filter-group">
                    <label>From Date</label>
                    <input type="date" name="date_from" value="<?=h($from)?>">
                </div>
                
                <div class="filter-group">
                    <label>To Date</label>
                    <input type="date" name="date_to" value="<?=h($to)?>">
                </div>
                
                <div class="filter-group">
                    <label>Success Status</label>
                    <select name="success_filter">
                        <option value="">All Status</option>
                        <?php foreach ($success_options as $option): ?>
                            <option value="<?=h($option)?>" <?=($success_filter === $option) ? 'selected' : ''?>>
                                <?=h($option)?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="filter-group">
                    <label>Call Type</label>
                    <select name="call_type_filter">
                        <option value="">All Types</option>
                        <?php foreach ($call_type_options as $option): ?>
                            <option value="<?=h($option)?>" <?=($call_type_filter === $option) ? 'selected' : ''?>>
                                <?=h(ucfirst($option))?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="filter-group">
                    <label>Per Page</label>
                    <select name="per_page">
                        <option value="25"  <?=$perPage===25?'selected':''?>>25</option>
                        <option value="50"  <?=$perPage===50?'selected':''?>>50</option>
                        <option value="100" <?=$perPage===100?'selected':''?>>100</option>
                        <option value="200" <?=$perPage===200?'selected':''?>>200</option>
                    </select>
                </div>
                
                <div class="filter-actions">
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-search"></i> Apply Filters
                    </button>
                    <a href="?export=xlsx<?= $whereSql ? '&' . http_build_query(array_filter($_GET, fn($k) => $k !== 'export' && $k !== 'page', ARRAY_FILTER_USE_KEY)) : '' ?>" 
                       class="btn btn-success">
                        <i class="fas fa-file-excel"></i> Export to Excel
                    </a>
                    <a href="<?= $_SERVER['PHP_SELF'] ?>" class="btn btn-secondary">
                        <i class="fas fa-redo"></i> Reset Filters
                    </a>
                </div>
            </form>
            
            <?php if (!empty($active_filters)): ?>
            <div class="filter-summary">
                <strong style="color: #2d3748;"><i class="fas fa-info-circle"></i> Active Filters:</strong>
                <?php foreach ($active_filters as $filter): ?>
                    <span class="filter-tag"><?= h($filter) ?></span>
                <?php endforeach; ?>
                <a href="<?= $_SERVER['PHP_SELF'] ?>" 
                   style="margin-left: auto; color: #e53e3e; text-decoration: none; font-weight: 600;">
                    <i class="fas fa-times"></i> Clear All
                </a>
            </div>
            <?php endif; ?>
            
            <div class="stats-bar">
                <div class="stats-info">
                    <span class="stats-highlight"><?=number_format($total)?></span> total calls found
                    <?php if (!empty($active_filters)): ?>
                        <span style="color: #3182ce; font-weight: 600;"> (filtered)</span>
                    <?php endif; ?>
                </div>
                <div class="stats-info">
                    Page <span class="stats-highlight"><?=$page?></span> of <span class="stats-highlight"><?=max(1,ceil($total/$perPage))?></span>
                </div>
            </div>

            <?php if (count($rows) > 0): ?>
            <div class="table-container">
                <table>
                    <thead>
                        <tr>
                            <th><i class="fas fa-id-card"></i> Call ID</th>
                            <th><i class="fas fa-robot"></i> Assistant</th>
                            <th><i class="fas fa-phone"></i> Assistant Phone</th>
                            <th><i class="fas fa-user"></i> Customer Phone</th>
                            <th><i class="fas fa-exchange-alt"></i> Type</th>
                            <th><i class="fas fa-info-circle"></i> End Reason</th>
                            <th><i class="fas fa-chart-line"></i> Success</th>
                            <th><i class="fas fa-clock"></i> Start Time</th>
                            <th><i class="fas fa-hourglass-half"></i> Duration</th>
                        </tr>
                    </thead>
                    <tbody>
                    <?php foreach ($rows as $r): ?>
                        <tr>
                            <td>
                                <a href="call_details.php?call_id=<?=urlencode($r['call_id'])?>" class="call-id">
                                    <?=h(substr($r['call_id'], 0, 8))?>...
                                </a>
                            </td>
                            <td><strong><?=h($r['assistant_name'] ?: 'Unknown')?></strong></td>
                            <td><?=h($r['assistant_phone'] ?: 'N/A')?></td>
                            <td><?=h($r['customer_phone'] ?: 'N/A')?></td>
                            <td>
                                <span class="pill <?=strtolower($r['call_type'])?>">
                                    <?=h($r['call_type'] ?: 'Unknown')?>
                                </span>
                            </td>
                            <td><?=h($r['ended_reason'] ?: 'N/A')?></td>
                            <td>
                                <?php 
                                $success = strtolower((string)$r['success_eval']);
                                $isSuccess = in_array($success, ['success', 'successful', 'completed']);
                                ?>
                                <span class="pill <?=$isSuccess ? 'success' : 'fail'?>">
                                    <?=h($r['success_eval'] ?: 'Unknown')?>
                                </span>
                            </td>
                            <td>
                                <?= $r['start_time'] ? date('M j, H:i', strtotime($r['start_time'])) : 'N/A' ?>
                            </td>
                            <td class="duration"><?= fmt_duration($r['duration_seconds']) ?: 'N/A' ?></td>
                        </tr>
                    <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            <?php else: ?>
            <div class="empty-state">
                <i class="fas fa-search"></i>
                <h3>No calls found</h3>
                <p>Try adjusting your search criteria or date range.</p>
            </div>
            <?php endif; ?>

            <?php
            $pages = max(1, (int)ceil($total / $perPage));
            if ($pages > 1):
                $qs = $_GET; 
                unset($qs['page']);
            ?>
            <div class="pagination">
                <?php if ($page > 1): 
                    $qs['page'] = $page - 1; ?>
                    <a href="?<?=http_build_query($qs)?>">
                        <i class="fas fa-chevron-left"></i> Previous
                    </a>
                <?php endif; ?>
                
                <span class="current">
                    <i class="fas fa-file-alt"></i> Page <?=$page?>
                </span>
                
                <?php if ($page < $pages): 
                    $qs['page'] = $page + 1; ?>
                    <a href="?<?=http_build_query($qs)?>">
                        Next <i class="fas fa-chevron-right"></i>
                    </a>
                <?php endif; ?>
            </div>
            <?php endif; ?>
        </div>
    </div>
</body>
</html>