<?php
require_once __DIR__ . '/vapi_config.php';

// Helper functions
function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
function fmt_duration($s){
  if ($s === null || $s === '') return '';
  $s = (int)$s; $m = intdiv($s, 60); $r = $s % 60;
  if ($m > 0) return $m . 'm ' . $r . 's';
  return $r . 's';
}
function fmt_money($x){
  return ($x === null || $x === '') ? '' : ('$' . number_format((float)$x, 2));
}

$call_id = $_GET['call_id'] ?? '';

if (!$call_id) {
    header('Location: call_logs.php');
    exit;
}

$pdo = new PDO(DB_DSN, DB_USER, DB_PASS, [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]);

// Get call details
$stmt = $pdo->prepare("SELECT * FROM vapi_call_logs WHERE call_id = ?");
$stmt->execute([$call_id]);
$call = $stmt->fetch(PDO::FETCH_ASSOC);

// Get transcript
$stmt = $pdo->prepare("
    SELECT * FROM vapi_call_transcripts 
    WHERE call_id = ? 
    ORDER BY sequence_number ASC
");
$stmt->execute([$call_id]);
$transcript = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get recording
$stmt = $pdo->prepare("SELECT * FROM vapi_call_recordings WHERE call_id = ?");
$stmt->execute([$call_id]);
$recording = $stmt->fetch(PDO::FETCH_ASSOC);

// Get AI analysis
$stmt = $pdo->prepare("SELECT * FROM vapi_call_analysis WHERE call_id = ?");
$stmt->execute([$call_id]);
$analysis = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$call) {
    echo "Call not found";
    exit;
}

// Helper function for sentiment color
function getSentimentColor($score) {
    if ($score === null) return '#6b7280';
    if ($score >= 0.5) return '#16a34a';
    if ($score >= 0.1) return '#eab308';
    if ($score >= -0.1) return '#f97316';
    return '#dc2626';
}

// Helper function for rating stars
function renderStars($rating) {
    if ($rating === null) return 'N/A';
    $stars = '';
    for ($i = 1; $i <= 10; $i++) {
        if ($i <= $rating) {
            $stars .= '<span style="color: #fbbf24;">★</span>';
        } else {
            $stars .= '<span style="color: #e5e7eb;">★</span>';
        }
    }
    return $stars . " ($rating/10)";
}
?>
<!doctype html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <title>Call Details - <?=h(substr($call_id, 0, 8))?>...</title>
    <meta name="viewport" content="width=device-width,initial-scale=1">
    <style>
        body{font-family:system-ui,-apple-system,Segoe UI,Roboto,Arial,sans-serif;background:#f8fafc;margin:0;padding:24px;}
        .container{max-width:1200px;margin:0 auto;}
        .card{background:#fff;border:1px solid #e5e7eb;border-radius:12px;padding:24px;margin-bottom:24px;box-shadow:0 1px 2px rgba(0,0,0,.04)}
        h1{margin:0 0 24px;font-size:24px;color:#111827;}
        h2{margin:0 0 16px;font-size:18px;color:#374151;border-bottom:2px solid #e5e7eb;padding-bottom:8px;}
        .header-actions{display:flex;justify-content:space-between;align-items:center;margin-bottom:24px;}
        .btn{display:inline-flex;align-items:center;padding:8px 16px;border:1px solid #d1d5db;border-radius:6px;text-decoration:none;color:#374151;background:#fff;font-size:14px;}
        .btn:hover{background:#f9fafb;}
        .btn-primary{background:#3b82f6;color:#fff;border-color:#3b82f6;}
        .btn-primary:hover{background:#2563eb;}
        
        .summary-grid{display:grid;grid-template-columns:repeat(auto-fit,minmax(200px,1fr));gap:16px;margin-bottom:24px;}
        .summary-item{background:#f8fafc;padding:16px;border-radius:8px;border-left:4px solid #3b82f6;}
        .summary-label{font-size:12px;color:#6b7280;text-transform:uppercase;font-weight:600;margin-bottom:4px;}
        .summary-value{font-size:16px;font-weight:600;color:#111827;}
        
        .analysis-grid{display:grid;grid-template-columns:1fr 1fr;gap:24px;margin-bottom:24px;}
        .analysis-card{background:#f8fafc;border:1px solid #e5e7eb;border-radius:8px;padding:20px;}
        
        .sentiment-bar{height:8px;background:#e5e7eb;border-radius:4px;overflow:hidden;margin:8px 0;}
        .sentiment-fill{height:100%;transition:width 0.3s ease;}
        
        .tags{display:flex;flex-wrap:wrap;gap:8px;margin-top:8px;}
        .tag{background:#e0f2fe;color:#0369a1;padding:4px 8px;border-radius:4px;font-size:12px;}
        
        .transcript-container{background:#fff;border:1px solid #e5e7eb;border-radius:8px;max-height:600px;overflow-y:auto;}
        .transcript-entry{padding:16px;border-bottom:1px solid #f3f4f6;}
        .transcript-entry:last-child{border-bottom:none;}
        .transcript-entry.assistant{background:#f0f9ff;}
        .transcript-entry.user{background:#f9fafb;}
        .transcript-speaker{font-weight:600;margin-bottom:8px;font-size:14px;}
        .transcript-speaker.assistant{color:#0369a1;}
        .transcript-speaker.user{color:#059669;}
        .transcript-message{line-height:1.5;color:#374151;}
        .transcript-time{font-size:12px;color:#9ca3af;margin-left:8px;}
        
        .recording-player{background:#f8fafc;border:1px solid #e5e7eb;border-radius:8px;padding:20px;text-align:center;}
        
        .no-data{text-align:center;color:#6b7280;padding:40px;font-style:italic;}
        
        .status-badge{display:inline-flex;padding:4px 8px;border-radius:12px;font-size:12px;font-weight:600;}
        .status-success{background:#ecfdf5;color:#065f46;border:1px solid #a7f3d0;}
        .status-fail{background:#fef2f2;color:#991b1b;border:1px solid:#fecaca;}
        .status-neutral{background:#f3f4f6;color:#374151;border:1px solid:#d1d5db;}
        
        .analyze-btn{background:#16a34a;color:#fff;border:none;padding:12px 24px;border-radius:6px;cursor:pointer;font-size:14px;}
        .analyze-btn:hover{background:#15803d;}
        .analyze-btn:disabled{background:#9ca3af;cursor:not-allowed;}
    </style>
</head>
<body>
    <div class="container">
        <div class="header-actions">
            <h1>Call Details</h1>
            <div>
                <?php if (!$analysis && !empty($transcript)): ?>
                <button onclick="analyzeCall('<?=h($call_id)?>')" class="analyze-btn" id="analyzeBtn">
                    Analyze Call with AI
                </button>
                <?php endif; ?>
                <a href="call_logs.php" class="btn">← Back to Call Logs</a>
            </div>
        </div>

        <!-- Call Summary -->
        <div class="card">
            <h2>Call Summary</h2>
            <div class="summary-grid">
                <div class="summary-item">
                    <div class="summary-label">Call ID</div>
                    <div class="summary-value" style="font-size:12px;font-family:monospace;"><?=h($call['call_id'])?></div>
                </div>
                <div class="summary-item">
                    <div class="summary-label">Assistant</div>
                    <div class="summary-value"><?=h($call['assistant_name'])?></div>
                </div>
                <div class="summary-item">
                    <div class="summary-label">Duration</div>
                    <div class="summary-value"><?=fmt_duration($call['duration_seconds'])?></div>
                </div>
                <div class="summary-item">
                    <div class="summary-label">Cost</div>
                    <div class="summary-value"><?=fmt_money($call['cost_usd'])?></div>
                </div>
                <div class="summary-item">
                    <div class="summary-label">Start Time</div>
                    <div class="summary-value"><?= $call['start_time'] ? date('M j, Y H:i', strtotime($call['start_time'])) : 'N/A' ?></div>
                </div>
                <div class="summary-item">
                    <div class="summary-label">Status</div>
                    <div class="summary-value">
                        <?php $success = strtolower($call['success_eval'] ?? '') === 'success'; ?>
                        <span class="status-badge <?=$success ? 'status-success' : 'status-fail'?>">
                            <?=h($call['success_eval'] ?? 'Unknown')?>
                        </span>
                    </div>
                </div>
                <div class="summary-item">
                    <div class="summary-label">End Reason</div>
                    <div class="summary-value"><?=h($call['ended_reason'] ?? 'N/A')?></div>
                </div>
                <div class="summary-item">
                    <div class="summary-label">Phone Numbers</div>
                    <div class="summary-value" style="font-size:12px;">
                        Customer: <?=h($call['customer_phone'] ?? 'N/A')?><br>
                        Assistant: <?=h($call['assistant_phone'] ?? 'N/A')?>
                    </div>
                </div>
            </div>
        </div>

        <!-- AI Analysis -->
        <?php if ($analysis): ?>
        <div class="card">
            <h2>AI Analysis</h2>
            <div class="analysis-grid">
                <div class="analysis-card">
                    <h3 style="margin-top:0;">Sentiment Analysis</h3>
                    <div>Score: <strong><?=number_format($analysis['sentiment_score'] ?? 0, 2)?></strong></div>
                    <div class="sentiment-bar">
                        <div class="sentiment-fill" style="width:<?=((($analysis['sentiment_score'] ?? 0) + 1) * 50)?>%;background:<?=getSentimentColor($analysis['sentiment_score'])?>"></div>
                    </div>
                    <div style="font-size:12px;color:#6b7280;">(-1.0 = Very Negative, 0 = Neutral, +1.0 = Very Positive)</div>
                </div>
                
                <div class="analysis-card">
                    <h3 style="margin-top:0;">Customer Satisfaction</h3>
                    <div><?=renderStars($analysis['customer_satisfaction'])?></div>
                </div>
            </div>
            
            <?php if ($analysis['key_topics']): ?>
            <div style="margin-bottom:20px;">
                <strong>Key Topics:</strong>
                <div class="tags">
                    <?php foreach (explode(', ', $analysis['key_topics']) as $topic): ?>
                        <span class="tag"><?=h($topic)?></span>
                    <?php endforeach; ?>
                </div>
            </div>
            <?php endif; ?>
            
            <?php if ($analysis['action_items']): ?>
            <div style="margin-bottom:20px;">
                <strong>Action Items:</strong>
                <ul>
                    <?php foreach (explode(', ', $analysis['action_items']) as $item): ?>
                        <li><?=h($item)?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
            <?php endif; ?>
            
            <?php if ($analysis['analysis_notes']): ?>
            <div>
                <strong>Analysis Notes:</strong>
                <p style="background:#f8fafc;padding:16px;border-radius:6px;margin-top:8px;"><?=nl2br(h($analysis['analysis_notes']))?></p>
            </div>
            <?php endif; ?>
        </div>
        <?php endif; ?>

        <!-- Recording -->
        <?php if ($recording && $recording['recording_url']): ?>
        <div class="card">
            <h2>Call Recording</h2>
            <div class="recording-player">
                <?php if ($recording['local_file_path'] && file_exists($recording['local_file_path'])): ?>
                <audio controls style="width:100%;max-width:500px;">
                    <source src="<?=h($recording['local_file_path'])?>" type="audio/wav">
                    <source src="<?=h($recording['local_file_path'])?>" type="audio/mp3">
                    Your browser does not support audio playback.
                </audio>
                <?php else: ?>
                <p>Recording available: <a href="<?=h($recording['recording_url'])?>" target="_blank">Play Recording</a></p>
                <p><small>Status: <?=h($recording['download_status'])?></small></p>
                <?php endif; ?>
            </div>
        </div>
        <?php endif; ?>

        <!-- Transcript -->
        <div class="card">
            <h2>Call Transcript</h2>
            <?php if ($transcript): ?>
            <div class="transcript-container">
                <?php foreach ($transcript as $entry): ?>
                <div class="transcript-entry <?=h($entry['role'])?>">
                    <div class="transcript-speaker <?=h($entry['role'])?>">
                        <?=ucfirst(h($entry['role']))?>
                        <?php if ($entry['timestamp']): ?>
                        <span class="transcript-time"><?=date('H:i:s', strtotime($entry['timestamp']))?></span>
                        <?php endif; ?>
                    </div>
                    <div class="transcript-message"><?=nl2br(h($entry['message']))?></div>
                </div>
                <?php endforeach; ?>
            </div>
            <?php else: ?>
            <div class="no-data">No transcript available for this call.</div>
            <?php endif; ?>
        </div>
    </div>

    <script>
    function analyzeCall(callId) {
        const btn = document.getElementById('analyzeBtn');
        btn.disabled = true;
        btn.textContent = 'Analyzing...';
        
        fetch('analyze_calls.php', {
            method: 'POST',
            headers: {'Content-Type': 'application/x-www-form-urlencoded'},
            body: `action=analyze_call&call_id=${callId}`
        })
        .then(response => response.json())
        .then(data => {
            if (data.error) {
                alert('Analysis failed: ' + data.error);
                btn.disabled = false;
                btn.textContent = 'Analyze Call with AI';
            } else {
                location.reload(); // Refresh to show analysis results
            }
        })
        .catch(error => {
            alert('Analysis failed: ' + error);
            btn.disabled = false;
            btn.textContent = 'Analyze Call with AI';
        });
    }
    </script>
</body>
</html>