<?php
// call_analyzer.php - Enhanced automatic call transcript analysis using Claude API
// ENHANCED: Better main/sub status detection and accurate follow-up date extraction
require_once __DIR__ . '/vapi_config.php';

class CallAnalyzer {
    private $pdo;
    private $openai_api_key;
    
    public function __construct() {
        try {
            $this->pdo = new PDO(DB_DSN, DB_USER, DB_PASS, [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4"
            ]);
            
            // Add your ChatGPT API key here
            $this->openai_api_key = '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';
            
            // Create analysis table if it doesn't exist
            $this->createAnalysisTable();
            
        } catch (Exception $e) {
            error_log('CallAnalyzer init failed: ' . $e->getMessage());
            throw $e;
        }
    }
    
    private function createAnalysisTable() {
        $sql = "CREATE TABLE IF NOT EXISTS vapi_call_analysiss (
            id INT AUTO_INCREMENT PRIMARY KEY,
            call_id VARCHAR(255) UNIQUE NOT NULL,
            
            -- AI Analysis Fields (Arabic)
            customer_satisfaction ENUM('راضي جداً', 'راضي', 'محايد', 'غير راضي', 'غير راضي جداً') DEFAULT 'محايد',
            call_result ENUM('نجح', 'نجح جزئياً', 'فشل', 'يحتاج متابعة') DEFAULT 'يحتاج متابعة',
            next_action TEXT,
            follow_up_date DATE NULL,
            should_call_again BOOLEAN DEFAULT FALSE,
            customer_needs TEXT,
            agent_performance ENUM('ممتاز', 'جيد', 'مقبول', 'يحتاج تحسين') DEFAULT 'مقبول',
            key_points TEXT,
            analysis_summary TEXT,
            raw_analysis JSON,
            
            -- Company-Specific Status Fields
            main_status ENUM(
                'لم يتم الوصول للعميل', 'تمت متابعة العميل', 'العميل وعد بالسداد', 
                'مغلق', 'العميل رافض السداد', 'خروج نهائي من المملكة', 
                'تم السداد', 'العميل متوفي', 'العميل مسجون', 'تسوية', 
                'خارج المملكة', 'وعد مكسور', 'العميل يتهرب من الاتصال', 
                'تم التواصل مع قريب أو صديق', 'العميل غير منتظم بالسداد', 
                'الاتصال علي الرقم الموحد', 'العميل طلب الاتصال لاحقا', 
                'غير معترف بالمديونية', 'الورثة واعدين باسداد', 'تم التواصل واتساب', 
                'الورثة يرفضون السداد', 'لا يرد', 'منتظم بالسداد شهريا', 'حالة غير محددة'
            ) NULL,
            
            sub_status ENUM(
                'الرقم لا يخص العميل', 'الرقم مقطوع', 'الرقم مغلق', 'لم يتم الرد', 
                'كامل المبلغ', 'سداد جزئي', 'اقساط', 'معترف بالمديونية ورافض السداد', 
                'لم يتم العثور علي رقم للتواصل', 'العميل يرد بدون وعد بالسداد', 
                'طلب مهلة للسداد', 'كامل المديونية', 'طلب اعفاء جزء من المديونية', 
                'معترض علي مبلغ المديونية', 'سدد دفعه الشهر', 'كامل', 
                'سدد كامل المديونية بعد الخصم', 'حالة فرعية غير محددة'
            ) NULL,
            
            -- Timestamps
            analyzed_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            
            FOREIGN KEY (call_id) REFERENCES vapi_call_logs(call_id) ON DELETE CASCADE,
            INDEX idx_call_id (call_id),
            INDEX idx_analyzed_at (analyzed_at),
            INDEX idx_customer_satisfaction (customer_satisfaction),
            INDEX idx_call_result (call_result),
            INDEX idx_main_status (main_status),
            INDEX idx_sub_status (sub_status)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        $this->pdo->exec($sql);
    }
    
    public function analyzeNewCalls() {
        // Find ALL calls that haven't been analyzed yet
        $sql = "
            SELECT DISTINCT cl.call_id, cl.assistant_name, cl.customer_phone, 
                   cl.start_time, cl.duration_seconds, cl.call_type, cl.success_eval
            FROM vapi_call_logs cl
            LEFT JOIN vapi_call_analysiss ca ON cl.call_id = ca.call_id
            WHERE ca.call_id IS NULL
            AND cl.duration_seconds > 1
            ORDER BY cl.start_time DESC
            LIMIT 10
        ";
        
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute();
        $calls = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $results = [];
        foreach ($calls as $call) {
            try {
                $analysis = $this->analyzeCall($call['call_id']);
                $results[] = [
                    'call_id' => $call['call_id'],
                    'status' => 'success',
                    'analysis' => $analysis
                ];
                
                // Wait between API calls to avoid rate limits
                sleep(2);
                
            } catch (Exception $e) {
                error_log("Failed to analyze call {$call['call_id']}: " . $e->getMessage());
                $results[] = [
                    'call_id' => $call['call_id'],
                    'status' => 'error',
                    'error' => $e->getMessage()
                ];
            }
        }
        
        return $results;
    }
    
    public function analyzeCall($call_id) {
        // Get call details
        $call = $this->getCallDetails($call_id);
        if (!$call) {
            throw new Exception("Call not found: $call_id");
        }
        
        // Get transcript
        $transcript = $this->getTranscript($call_id);
        
        // Handle calls with NO transcript
        if (empty($transcript)) {
            return $this->analyzeCallWithoutTranscript($call);
        }
        
        // Format transcript for analysis
        $formatted_transcript = $this->formatTranscript($transcript);
        
        // Analyze with ChatGPT API
        $analysis = $this->performChatGPTAnalysis($formatted_transcript, $call);
        
        // Ensure both AI analysis and company statuses are included
        $complete_analysis = [
            // AI Analysis Fields
            'customer_satisfaction' => $analysis['customer_satisfaction'],
            'call_result' => $analysis['call_result'],
            'next_action' => $analysis['next_action'],
            'follow_up_date' => $analysis['follow_up_date'],
            'should_call_again' => $analysis['should_call_again'],
            'customer_needs' => $analysis['customer_needs'],
            'agent_performance' => $analysis['agent_performance'],
            'key_points' => $analysis['key_points'],
            'analysis_summary' => $analysis['analysis_summary'],
            
            // Company-Specific Status Fields
            'main_status' => $analysis['main_status'],
            'sub_status' => $analysis['sub_status'],
            
            // Raw data for debugging
            'raw_analysis' => $analysis,
            'call_details' => $call
        ];
        
        // Save complete analysis to database
        $this->saveAnalysis($call_id, $complete_analysis);
        
        return $complete_analysis;
    }
    
    // Handle calls with no transcript at all
    private function analyzeCallWithoutTranscript($call) {
        $duration = $call['duration_seconds'] ?? 0;
        $ended_reason = $call['ended_reason'] ?? '';
        
        // Basic analysis for calls without transcript
        $analysis = [
            'customer_satisfaction' => 'محايد',
            'call_result' => 'فشل',
            'next_action' => 'إعادة محاولة الاتصال',
            'follow_up_date' => date('Y-m-d', strtotime('+1 day')),
            'should_call_again' => true,
            'customer_needs' => 'غير واضح - لم يتم تسجيل محادثة',
            'agent_performance' => 'مقبول',
            'key_points' => 'مكالمة بدون نص محادثة',
            'analysis_summary' => "مكالمة مدتها {$duration} ثانية انتهت بسبب: {$ended_reason}. لم يتم تسجيل نص محادثة.",
            'main_status' => 'لم يتم الوصول للعميل',
            'sub_status' => $duration < 10 ? 'لم يتم الرد' : 'حالة فرعية غير محددة',
            'raw_analysis' => [
                'note' => 'analyzed_without_transcript',
                'duration' => $duration,
                'ended_reason' => $ended_reason
            ]
        ];
        
        return $analysis;
    }
    
    private function getCallDetails($call_id) {
        $stmt = $this->pdo->prepare("SELECT * FROM vapi_call_logs WHERE call_id = ?");
        $stmt->execute([$call_id]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    private function getTranscript($call_id) {
        $stmt = $this->pdo->prepare("
            SELECT role, message, timestamp, sequence_number 
            FROM vapi_call_transcripts 
            WHERE call_id = ? 
            ORDER BY sequence_number ASC
        ");
        $stmt->execute([$call_id]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    // Enhanced transcript formatting with short call detection
    private function formatTranscript($transcript) {
        $formatted = "";
        $total_chars = 0;
        $message_count = 0;
        
        foreach ($transcript as $entry) {
            $role = ($entry['role'] == 'assistant') ? 'الوكيل' : 'العميل';
            $message = trim($entry['message']);
            if (!empty($message)) {
                $formatted .= "$role: $message\n";
                $total_chars += strlen($message);
                $message_count++;
            }
        }
        
        // If transcript is very short, add context
        if ($message_count <= 2 || $total_chars < 50) {
            $formatted .= "\n[ملاحظة: مكالمة قصيرة أو غير مكتملة - $message_count رسائل، $total_chars حرف]";
        }
        
        return $formatted;
    }
    
    private function performChatGPTAnalysis($transcript, $call_details) {
        $prompt = $this->buildAnalysisPrompt($transcript, $call_details);
        
        $response = $this->callChatGPT($prompt);
        
        if (isset($response['error'])) {
            throw new Exception($response['error']);
        }
        
        // Parse ChatGPT response
        return $this->parseChatGPTResponse($response['choices'][0]['message']['content']);
    }
    
    // ENHANCED: Better prompt for accurate status and date detection
    private function buildAnalysisPrompt($transcript, $call_details) {
        $current_date = date('Y-m-d');
        $current_year = date('Y');
        
        return "أنت محلل مكالمات خدمة عملاء محترف متخصص في تحصيل الديون. حلل هذه المكالمة واعطني تقييم مفصل.

**تفاصيل المكالمة:**
- المدة: " . ($call_details['duration_seconds'] ?? 0) . " ثانية
- نوع المكالمة: " . ($call_details['call_type'] ?? 'غير محدد') . "
- الوكيل: " . ($call_details['assistant_name'] ?? 'غير محدد') . "
- التقييم الأولي: " . ($call_details['success_eval'] ?? 'غير محدد') . "
- تاريخ اليوم: $current_date

**نص المحادثة:**
$transcript

**تعليمات مهمة للتحليل:**

1. **للحالة الرئيسية (main_status):**
   - استخدم 'تمت متابعة العميل' فقط إذا تم التواصل مع العميل الفعلي وتم الحديث معه
   - إذا لم يرد العميل أو تم التحدث مع شخص آخر، استخدم 'لم يتم الوصول للعميل'
   - إذا أغلق العميل الخط فور سماع الوكيل، استخدم 'العميل يتهرب من الاتصال'
   - إذا كان الرقم خطأ أو لا يخص العميل، استخدم 'لم يتم الوصول للعميل'

2. **لتاريخ المتابعة (follow_up_date):**
   - ابحث في المحادثة عن أي تاريخ محدد ذكره العميل أو الوكيل
   - إذا قال العميل 'الأسبوع الجاي' أو 'الأسبوع القادم'، احسب التاريخ بعد 7 أيام من اليوم: " . date('Y-m-d', strtotime('+7 days')) . "
   - إذا قال 'الشهر الجاي' أو 'الشهر القادم'، احسب التاريخ بعد شهر من اليوم
   - إذا ذكر يوم معين مثل 'يوم الأحد' أو 'الثلاثاء'، احسب أقرب تاريخ لهذا اليوم
   - إذا ذكر تاريخ بدون سنة، افترض أنه في السنة الحالية ($current_year)
   - إذا لم يذكر تاريخ محدد ولكن وعد بالاتصال، استخدم تاريخ بعد 3 أيام
   - إذا لم يتم التواصل مع العميل، استخدم تاريخ بعد يوم واحد

3. **كن دقيقاً في:**
   - التمييز بين التحدث مع العميل نفسه أو مع شخص آخر
   - استخراج التواريخ المذكورة في المحادثة وتحويلها لصيغة صحيحة
   - تحديد ما إذا كان العميل وعد فعلاً بالسداد أم لا

للمكالمات القصيرة أو غير المكتملة:
- إذا لم يرد العميل أو أغلق الخط: استخدم 'لم يتم الوصول للعميل'
- إذا كان الرقم خطأ أو لا يخص العميل: استخدم 'الرقم لا يخص العميل'
- إذا كان الرقم مقطوع: استخدم 'الرقم مقطوع'
- إذا أجاب شخص آخر: استخدم 'تم التواصل مع قريب أو صديق'

قم بتحليل المكالمة وأرجع النتيجة في صيغة JSON بالضبط كما يلي (بالللغة العربية):

{
    \"customer_satisfaction\": \"راضي\" أو \"غير راضي\" أو \"محايد\" أو \"راضي جداً\" أو \"غير راضي جداً\",
    \"call_result\": \"نجح\" أو \"نجح جزئياً\" أو \"فشل\" أو \"يحتاج متابعة\",
    \"next_action\": \"وصف للإجراء المطلوب التالي\",
    \"follow_up_date\": \"YYYY-MM-DD\" أو null,
    \"should_call_again\": true أو false,
    \"customer_needs\": \"احتياجات العميل المحددة أو 'غير واضح' للمكالمات القصيرة\",
    \"agent_performance\": \"ممتاز\" أو \"جيد\" أو \"مقبول\" أو \"يحتاج تحسين\",
    \"key_points\": \"النقاط الرئيسية في المكالمة أو 'مكالمة قصيرة'\",
    \"analysis_summary\": \"ملخص شامل للمكالمة والتوصيات\",
    \"main_status\": \"الحالة الرئيسية المقترحة من القائمة أدناه\",
    \"sub_status\": \"الحالة الفرعية المقترحة من القائمة أدناه\"
}

**الحالات الرئيسية المتاحة:**
- لم يتم الوصول للعميل (للمكالمات التي لم يرد فيها العميل)
- تمت متابعة العميل (فقط عند التحدث مع العميل الفعلي)
- العميل وعد بالسداد
- مغلق
- العميل رافض السداد
- خروج نهائي من المملكة
- تم السداد
- العميل متوفي
- العميل مسجون
- تسوية
- خارج المملكة
- وعد مكسور
- العميل يتهرب من الاتصال
- تم التواصل مع قريب أو صديق
- العميل غير منتظم بالسداد
- الاتصال علي الرقم الموحد
- العميل طلب الاتصال لاحقا
- غير معترف بالمديونية
- الورثة واعدين باسداد
- تم التواصل واتساب
- الورثة يرفضون السداد
- لا يرد
- منتظم بالسداد شهريا
- حالة غير محددة

**الحالات الفرعية المتاحة:**
- الرقم لا يخص العميل
- الرقم مقطوع
- الرقم مغلق
- لم يتم الرد (للمكالمات التي لم يجب عليها أحد)
- كامل المبلغ
- سداد جزئي
- اقساط
- معترف بالمديونية ورافض السداد
- لم يتم العثور علي رقم للتواصل
- العميل يرد بدون وعد بالسداد
- طلب مهلة للسداد
- كامل المديونية
- طلب اعفاء جزء من المديونية
- معترض علي مبلغ المديونية
- سدد دفعه الشهر
- كامل
- سدد كامل المديونية بعد الخصم
- حالة فرعية غير محددة

تأكد من:
1. استخدام المعلومات المتاحة حتى لو كانت قليلة
2. اختيار أنسب الحالات للمكالمات القصيرة
3. تقديم توصيات عملية للمتابعة
4. التعامل مع حالات عدم الرد أو الأرقام الخطأ بطريقة مناسبة
5. حساب التواريخ بدقة بناءً على ما ذكر في المحادثة

أرجع JSON فقط، بدون أي نص إضافي.";
    }
    
    private function callChatGPT($prompt) {
        $data = [
            'model' => 'gpt-4',
            'messages' => [
                [
                    'role' => 'system',
                    'content' => 'أنت محلل مكالمات خدمة عملاء محترف متخصص في تحصيل الديون. تحلل المكالمات وتعطي تقييمات دقيقة ومفصلة باللغة العربية.'
                ],
                [
                    'role' => 'user',
                    'content' => $prompt
                ]
            ],
            'max_tokens' => 1500,
            'temperature' => 0.3
        ];
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, 'https://api.openai.com/v1/chat/completions');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_TIMEOUT, 60);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Authorization: Bearer ' . $this->openai_api_key
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlError = curl_error($ch);
        curl_close($ch);
        
        if ($curlError) {
            throw new Exception("ChatGPT API Error: " . $curlError);
        }
        
        if ($httpCode !== 200) {
            error_log("ChatGPT API Error: HTTP $httpCode - Response: $response");
            throw new Exception("ChatGPT API error: HTTP " . $httpCode);
        }
        
        $result = json_decode($response, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception("Failed to parse ChatGPT response");
        }
        
        return $result;
    }
    
    private function parseChatGPTResponse($response) {
        // Clean up the response (remove any markdown formatting)
        $response = trim($response);
        $response = preg_replace('/^```json\s*/', '', $response);
        $response = preg_replace('/\s*```$/', '', $response);
        
        $analysis = json_decode($response, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            error_log("JSON Parse Error: " . json_last_error_msg() . " | Response: " . $response);
            throw new Exception('Failed to parse ChatGPT response: ' . json_last_error_msg());
        }
        
        // Validate and set defaults
        return [
            'customer_satisfaction' => $this->validateSatisfaction($analysis['customer_satisfaction'] ?? 'محايد'),
            'call_result' => $this->validateCallResult($analysis['call_result'] ?? 'يحتاج متابعة'),
            'next_action' => $this->validateString($analysis['next_action'] ?? ''),
            'follow_up_date' => $this->validateDate($analysis['follow_up_date'] ?? null),
            'should_call_again' => $analysis['should_call_again'] ?? false,
            'customer_needs' => $this->validateString($analysis['customer_needs'] ?? ''),
            'agent_performance' => $this->validatePerformance($analysis['agent_performance'] ?? 'مقبول'),
            'key_points' => $this->validateString($analysis['key_points'] ?? ''),
            'analysis_summary' => $this->validateString($analysis['analysis_summary'] ?? ''),
            'main_status' => $this->validateMainStatus($analysis['main_status'] ?? null),
            'sub_status' => $this->validateSubStatus($analysis['sub_status'] ?? null),
            'raw_analysis' => $analysis
        ];
    }
    
    private function validateMainStatus($value) {
        $valid = [
            'لم يتم الوصول للعميل', 'تمت متابعة العميل', 'العميل وعد بالسداد', 
            'مغلق', 'العميل رافض السداد', 'خروج نهائي من المملكة', 
            'تم السداد', 'العميل متوفي', 'العميل مسجون', 'تسوية', 
            'خارج المملكة', 'وعد مكسور', 'العميل يتهرب من الاتصال', 
            'تم التواصل مع قريب أو صديق', 'العميل غير منتظم بالسداد', 
            'الاتصال علي الرقم الموحد', 'العميل طلب الاتصال لاحقا', 
            'غير معترف بالمديونية', 'الورثة واعدين باسداد', 'تم التواصل واتساب', 
            'الورثة يرفضون السداد', 'لا يرد', 'منتظم بالسداد شهريا', 'حالة غير محددة'
        ];
        return in_array($value, $valid) ? $value : null;
    }
    
    private function validateSubStatus($value) {
        $valid = [
            'الرقم لا يخص العميل', 'الرقم مقطوع', 'الرقم مغلق', 'لم يتم الرد', 
            'كامل المبلغ', 'سداد جزئي', 'اقساط', 'معترف بالمديونية ورافض السداد', 
            'لم يتم العثور علي رقم للتواصل', 'العميل يرد بدون وعد بالسداد', 
            'طلب مهلة للسداد', 'كامل المديونية', 'طلب اعفاء جزء من المديونية', 
            'معترض علي مبلغ المديونية', 'سدد دفعه الشهر', 'كامل', 
            'سدد كامل المديونية بعد الخصم', 'حالة فرعية غير محددة'
        ];
        return in_array($value, $valid) ? $value : null;
    }
    
    private function validateSatisfaction($value) {
        $valid = ['راضي جداً', 'راضي', 'محايد', 'غير راضي', 'غير راضي جداً'];
        return in_array($value, $valid) ? $value : 'محايد';
    }
    
    private function validateCallResult($value) {
        $valid = ['نجح', 'نجح جزئياً', 'فشل', 'يحتاج متابعة'];
        return in_array($value, $valid) ? $value : 'يحتاج متابعة';
    }
    
    private function validatePerformance($value) {
        $valid = ['ممتاز', 'جيد', 'مقبول', 'يحتاج تحسين'];
        return in_array($value, $valid) ? $value : 'مقبول';
    }
    
    private function validateString($value) {
        return is_string($value) ? trim($value) : '';
    }
    
    private function validateDate($value) {
        if (!$value || $value === 'null') return null;
        if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $value)) {
            return $value;
        }
        return null;
    }
    
    private function saveAnalysis($call_id, $analysis) {
        $sql = "INSERT INTO vapi_call_analysiss 
                (call_id, customer_satisfaction, call_result, next_action, follow_up_date,
                 should_call_again, customer_needs, agent_performance, key_points, 
                 analysis_summary, main_status, sub_status, raw_analysis)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE
                    customer_satisfaction = VALUES(customer_satisfaction),
                    call_result = VALUES(call_result),
                    next_action = VALUES(next_action),
                    follow_up_date = VALUES(follow_up_date),
                    should_call_again = VALUES(should_call_again),
                    customer_needs = VALUES(customer_needs),
                    agent_performance = VALUES(agent_performance),
                    key_points = VALUES(key_points),
                    analysis_summary = VALUES(analysis_summary),
                    main_status = VALUES(main_status),
                    sub_status = VALUES(sub_status),
                    raw_analysis = VALUES(raw_analysis),
                    updated_at = CURRENT_TIMESTAMP";
        
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([
            $call_id,
            $analysis['customer_satisfaction'],
            $analysis['call_result'],
            $analysis['next_action'],
            $analysis['follow_up_date'],
            $analysis['should_call_again'] ? 1 : 0,
            $analysis['customer_needs'],
            $analysis['agent_performance'],
            $analysis['key_points'],
            $analysis['analysis_summary'],
            $analysis['main_status'],
            $analysis['sub_status'],
            json_encode($analysis['raw_analysis'])
        ]);
    }
    
    // Get analysis by call ID
    public function getAnalysis($call_id) {
        $stmt = $this->pdo->prepare("
            SELECT ca.*, cl.assistant_name, cl.customer_phone, cl.start_time, 
                   cl.duration_seconds, cl.call_type, cl.ended_reason
            FROM vapi_call_analysiss ca
            JOIN vapi_call_logs cl ON ca.call_id = cl.call_id
            WHERE ca.call_id = ?
        ");
        $stmt->execute([$call_id]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$result) {
            return null;
        }
        
        return [
            'call_info' => [
                'call_id' => $result['call_id'],
                'assistant_name' => $result['assistant_name'],
                'customer_phone' => $result['customer_phone'],
                'start_time' => $result['start_time'],
                'duration_seconds' => $result['duration_seconds'],
                'call_type' => $result['call_type'],
                'ended_reason' => $result['ended_reason']
            ],
            'ai_analysis' => [
                'customer_satisfaction' => $result['customer_satisfaction'],
                'call_result' => $result['call_result'],
                'agent_performance' => $result['agent_performance'],
                'next_action' => $result['next_action'],
                'follow_up_date' => $result['follow_up_date'],
                'should_call_again' => $result['should_call_again'],
                'customer_needs' => $result['customer_needs'],
                'key_points' => $result['key_points'],
                'analysis_summary' => $result['analysis_summary']
            ],
            'company_status' => [
                'main_status' => $result['main_status'],
                'sub_status' => $result['sub_status']
            ],
            'timestamps' => [
                'analyzed_at' => $result['analyzed_at'],
                'updated_at' => $result['updated_at']
            ]
        ];
    }
    
    public function getAnalysisStats() {
        $sql = "SELECT 
                    COUNT(*) as total_analyzed,
                    COUNT(CASE WHEN customer_satisfaction IN ('راضي', 'راضي جداً') THEN 1 END) as satisfied_customers,
                    COUNT(CASE WHEN call_result = 'نجح' THEN 1 END) as successful_calls,
                    COUNT(CASE WHEN should_call_again = 1 THEN 1 END) as needs_followup,
                    COUNT(CASE WHEN agent_performance IN ('ممتاز', 'جيد') THEN 1 END) as good_performance,
                    COUNT(CASE WHEN main_status IS NOT NULL THEN 1 END) as has_company_status,
                    COUNT(CASE WHEN main_status = 'تم السداد' THEN 1 END) as payment_completed,
                    COUNT(CASE WHEN main_status = 'العميل وعد بالسداد' THEN 1 END) as payment_promised
                FROM vapi_call_analysiss";
        
        $stmt = $this->pdo->query($sql);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    // Get recent analyses for dashboard
    public function getRecentAnalyses($limit = 20) {
        $sql = "
            SELECT ca.*, cl.assistant_name, cl.customer_phone, cl.start_time, cl.duration_seconds
            FROM vapi_call_analysiss ca
            JOIN vapi_call_logs cl ON ca.call_id = cl.call_id
            ORDER BY ca.analyzed_at DESC
            LIMIT ?
        ";
        
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([$limit]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    // Get calls that need follow-up
    public function getCallsNeedingFollowUp() {
        $sql = "
            SELECT ca.*, cl.assistant_name, cl.customer_phone, cl.start_time
            FROM vapi_call_analysiss ca
            JOIN vapi_call_logs cl ON ca.call_id = cl.call_id
            WHERE ca.should_call_again = 1 
            AND (ca.follow_up_date IS NULL OR ca.follow_up_date <= CURDATE())
            ORDER BY ca.follow_up_date ASC, ca.analyzed_at DESC
            LIMIT 50
        ";
        
        $stmt = $this->pdo->query($sql);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    // Get unanalyzed calls
    public function getUnanalyzedCalls($limit = 20) {
        $sql = "
            SELECT cl.call_id, cl.assistant_name, cl.customer_phone, 
                   cl.start_time, cl.duration_seconds, cl.call_type
            FROM vapi_call_logs cl
            LEFT JOIN vapi_call_analysiss ca ON cl.call_id = ca.call_id
            WHERE ca.call_id IS NULL
            AND cl.duration_seconds > 1
            ORDER BY cl.start_time DESC
            LIMIT ?
        ";
        
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([$limit]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    // Handle web requests
    public function handleWebRequest() {
        $action = $_GET['action'] ?? '';
        
        header('Content-Type: application/json; charset=utf-8');
        
        try {
            switch ($action) {
                case 'analyze_new':
                    $results = $this->analyzeNewCalls();
                    echo json_encode(['success' => true, 'results' => $results], JSON_UNESCAPED_UNICODE);
                    break;
                    
                case 'analyze_single':
                    $call_id = $_GET['call_id'] ?? '';
                    if (!$call_id) {
                        throw new Exception('Call ID is required');
                    }
                    $analysis = $this->analyzeCall($call_id);
                    echo json_encode(['success' => true, 'analysis' => $analysis], JSON_UNESCAPED_UNICODE);
                    break;
                    
                case 'get_analysis':
                    $call_id = $_GET['call_id'] ?? '';
                    if (!$call_id) {
                        throw new Exception('Call ID is required');
                    }
                    $analysis = $this->getAnalysis($call_id);
                    echo json_encode(['success' => true, 'analysis' => $analysis], JSON_UNESCAPED_UNICODE);
                    break;
                    
                case 'get_stats':
                    $stats = $this->getAnalysisStats();
                    echo json_encode(['success' => true, 'stats' => $stats], JSON_UNESCAPED_UNICODE);
                    break;
                    
                case 'dashboard':
                    $this->showDashboard();
                    break;
                    
                default:
                    $this->showDashboard();
                    break;
            }
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'error' => $e->getMessage()], JSON_UNESCAPED_UNICODE);
        }
    }
    
    // Enhanced dashboard with better analytics
    private function showDashboard() {
        $stats = $this->getAnalysisStats();
        $recent_analyses = $this->getRecentAnalyses(10);
        $unanalyzed_calls = $this->getUnanalyzedCalls(10);
        $followup_calls = $this->getCallsNeedingFollowUp();
        
        ?><!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="utf-8">
    <title>محلل المكالمات المطور - ChatGPT AI</title>
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <style>
        * { box-sizing: border-box; }
        body {
            font-family: 'Segoe UI', Tahoma, Arial, sans-serif;
            margin: 0;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            direction: rtl;
        }
        .container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 20px;
        }
        h1 {
            color: white;
            text-align: center;
            margin-bottom: 30px;
            font-size: 2.5rem;
            text-shadow: 2px 2px 4px rgba(0,0,0,0.3);
        }
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        .stat-card {
            background: white;
            padding: 20px;
            border-radius: 15px;
            box-shadow: 0 8px 25px rgba(0,0,0,0.1);
            text-align: center;
            transition: transform 0.3s ease;
        }
        .stat-card:hover { transform: translateY(-5px); }
        .stat-number {
            font-size: 2.5rem;
            font-weight: bold;
            color: #4f46e5;
            margin-bottom: 10px;
        }
        .stat-label {
            color: #6b7280;
            font-size: 0.9rem;
        }
        .card {
            background: white;
            border-radius: 15px;
            padding: 25px;
            margin-bottom: 25px;
            box-shadow: 0 8px 25px rgba(0,0,0,0.1);
        }
        .card h2 {
            color: #374151;
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 15px;
        }
        th, td {
            padding: 12px;
            text-align: right;
            border-bottom: 1px solid #e5e7eb;
        }
        th {
            background: #f9fafb;
            font-weight: 600;
            color: #374151;
        }
        .btn {
            background: linear-gradient(135deg, #10b981 0%, #059669 100%);
            color: white;
            padding: 10px 20px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-size: 14px;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 8px;
            transition: all 0.3s ease;
        }
        .btn:hover { transform: translateY(-2px); }
        .btn-primary { background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%); }
        .btn-danger { background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%); }
        .btn-small { padding: 6px 12px; font-size: 12px; }
        .status-badge {
            padding: 4px 8px;
            border-radius: 6px;
            font-size: 12px;
            font-weight: 500;
        }
        .status-success { background: #dcfce7; color: #166534; }
        .status-warning { background: #fef3c7; color: #92400e; }
        .status-danger { background: #fee2e2; color: #991b1b; }
        .status-info { background: #dbeafe; color: #1e40af; }
        .actions {
            display: flex;
            gap: 15px;
            margin-bottom: 20px;
            flex-wrap: wrap;
        }
        .log {
            background: #1f2937;
            color: #e5e7eb;
            padding: 15px;
            border-radius: 8px;
            font-family: 'Courier New', monospace;
            font-size: 13px;
            max-height: 300px;
            overflow-y: auto;
            white-space: pre-wrap;
            margin-top: 15px;
        }
        .loading {
            opacity: 0.6;
            pointer-events: none;
        }
        @media (max-width: 768px) {
            .container { padding: 10px; }
            h1 { font-size: 1.8rem; }
            .stats-grid { grid-template-columns: repeat(2, 1fr); }
            .actions { flex-direction: column; }
        }
    </style>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
</head>
<body>
    <div class="container">
        <h1><i class="fas fa-brain"></i> محلل المكالمات المطور - ChatGPT AI</h1>
        
        <!-- Statistics -->
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-number"><?= number_format($stats['total_analyzed'] ?? 0) ?></div>
                <div class="stat-label">إجمالي المكالمات المحللة</div>
            </div>
            <div class="stat-card">
                <div class="stat-number"><?= number_format($stats['satisfied_customers'] ?? 0) ?></div>
                <div class="stat-label">عملاء راضون</div>
            </div>
            <div class="stat-card">
                <div class="stat-number"><?= number_format($stats['successful_calls'] ?? 0) ?></div>
                <div class="stat-label">مكالمات ناجحة</div>
            </div>
            <div class="stat-card">
                <div class="stat-number"><?= number_format($stats['needs_followup'] ?? 0) ?></div>
                <div class="stat-label">تحتاج متابعة</div>
            </div>
            <div class="stat-card">
                <div class="stat-number"><?= number_format($stats['payment_completed'] ?? 0) ?></div>
                <div class="stat-label">تم السداد</div>
            </div>
            <div class="stat-card">
                <div class="stat-number"><?= number_format($stats['payment_promised'] ?? 0) ?></div>
                <div class="stat-label">وعد بالسداد</div>
            </div>
        </div>
        
        <!-- Actions -->
        <div class="card">
            <h2><i class="fas fa-cogs"></i> إجراءات التحليل</h2>
            <div class="actions">
                <button class="btn" onclick="analyzeNewCalls()">
                    <i class="fas fa-play"></i> تحليل المكالمات الجديدة
                </button>
                <button class="btn btn-primary" onclick="loadOverview()">
                    <i class="fas fa-sync"></i> تحديث الإحصائيات
                </button>
                <a href="?action=dashboard" class="btn btn-primary">
                    <i class="fas fa-chart-bar"></i> تحديث الصفحة
                </a>
            </div>
            <div id="analysisLog" class="log" style="display: none;"></div>
        </div>
        
        <!-- Follow-up Calls -->
        <?php if (!empty($followup_calls)): ?>
        <div class="card">
            <h2><i class="fas fa-phone"></i> مكالمات تحتاج متابعة (<?= count($followup_calls) ?>)</h2>
            <div style="overflow-x: auto;">
                <table>
                    <thead>
                        <tr>
                            <th>معرف المكالمة</th>
                            <th>الوكيل</th>
                            <th>رقم العميل</th>
                            <th>الحالة الرئيسية</th>
                            <th>تاريخ المتابعة</th>
                            <th>إجراءات</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($followup_calls as $call): ?>
                        <tr>
                            <td><?= htmlspecialchars(substr($call['call_id'], 0, 8)) ?>...</td>
                            <td><?= htmlspecialchars($call['assistant_name']) ?></td>
                            <td><?= htmlspecialchars($call['customer_phone']) ?></td>
                            <td><?= htmlspecialchars($call['main_status'] ?? 'غير محدد') ?></td>
                            <td><?= $call['follow_up_date'] ? date('Y-m-d', strtotime($call['follow_up_date'])) : 'غير محدد' ?></td>
                            <td>
                                <button class="btn btn-small" onclick="getAnalysis('<?= $call['call_id'] ?>')">
                                    <i class="fas fa-eye"></i> عرض
                                </button>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
        <?php endif; ?>
        
        <!-- Unanalyzed Calls -->
        <?php if (!empty($unanalyzed_calls)): ?>
        <div class="card">
            <h2><i class="fas fa-exclamation-triangle"></i> مكالمات غير محللة (<?= count($unanalyzed_calls) ?>)</h2>
            <div style="overflow-x: auto;">
                <table>
                    <thead>
                        <tr>
                            <th>معرف المكالمة</th>
                            <th>الوكيل</th>
                            <th>رقم العميل</th>
                            <th>وقت المكالمة</th>
                            <th>المدة</th>
                            <th>إجراءات</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($unanalyzed_calls as $call): ?>
                        <tr>
                            <td><?= htmlspecialchars(substr($call['call_id'], 0, 8)) ?>...</td>
                            <td><?= htmlspecialchars($call['assistant_name']) ?></td>
                            <td><?= htmlspecialchars($call['customer_phone']) ?></td>
                            <td><?= $call['start_time'] ? date('M j, H:i', strtotime($call['start_time'])) : 'غير محدد' ?></td>
                            <td><?= $call['duration_seconds'] ? intval($call['duration_seconds']/60) . 'دقيقة' : 'غير محدد' ?></td>
                            <td>
                                <button class="btn btn-small btn-primary" onclick="analyzeSingleCall('<?= $call['call_id'] ?>')">
                                    <i class="fas fa-play"></i> تحليل الآن
                                </button>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
        <?php endif; ?>
        
        <!-- Recent Analyses -->
        <div class="card">
            <h2><i class="fas fa-history"></i> التحليلات الأخيرة</h2>
            <?php if (!empty($recent_analyses)): ?>
            <div style="overflow-x: auto;">
                <table>
                    <thead>
                        <tr>
                            <th>معرف المكالمة</th>
                            <th>الوكيل</th>
                            <th>رقم العميل</th>
                            <th>رضا العميل</th>
                            <th>نتيجة المكالمة</th>
                            <th>الحالة الرئيسية</th>
                            <th>أداء الوكيل</th>
                            <th>يحتاج متابعة</th>
                            <th>تاريخ التحليل</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($recent_analyses as $analysis): ?>
                        <tr>
                            <td><?= htmlspecialchars(substr($analysis['call_id'], 0, 8)) ?>...</td>
                            <td><?= htmlspecialchars($analysis['assistant_name']) ?></td>
                            <td><?= htmlspecialchars($analysis['customer_phone']) ?></td>
                            <td>
                                <span class="status-badge <?= strpos($analysis['customer_satisfaction'], 'راضي') !== false ? 'status-success' : 'status-warning' ?>">
                                    <?= htmlspecialchars($analysis['customer_satisfaction']) ?>
                                </span>
                            </td>
                            <td>
                                <span class="status-badge <?= $analysis['call_result'] == 'نجح' ? 'status-success' : 'status-danger' ?>">
                                    <?= htmlspecialchars($analysis['call_result']) ?>
                                </span>
                            </td>
                            <td><?= htmlspecialchars($analysis['main_status'] ?? 'غير محدد') ?></td>
                            <td><?= htmlspecialchars($analysis['agent_performance']) ?></td>
                            <td><?= $analysis['should_call_again'] ? '✅ نعم' : '❌ لا' ?></td>
                            <td><?= date('M j, H:i', strtotime($analysis['analyzed_at'])) ?></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            <?php else: ?>
            <p>لا توجد تحليلات متاحة حالياً.</p>
            <?php endif; ?>
        </div>
    </div>
    
    <script>
    function log(message) {
        const logDiv = document.getElementById('analysisLog');
        logDiv.style.display = 'block';
        logDiv.textContent += new Date().toLocaleTimeString() + ': ' + message + '\n';
        logDiv.scrollTop = logDiv.scrollHeight;
    }

    function loadOverview() {
        log('🔄 جاري تحديث الإحصائيات...');
        fetch('?action=get_stats')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                log('✅ تم تحديث الإحصائيات بنجاح');
                setTimeout(() => location.reload(), 1000);
            } else {
                log('❌ فشل تحديث الإحصائيات: ' + data.error);
            }
        })
        .catch(error => {
            log('❌ خطأ في تحديث الإحصائيات: ' + error);
        });
    }

    function analyzeNewCalls() {
        const btn = event.target;
        btn.disabled = true;
        btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> جاري التحليل...';
        
        log('🚀 بدء تحليل المكالمات الجديدة...');
        
        fetch('?action=analyze_new')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                log(`✅ تم تحليل ${data.results.length} مكالمة بنجاح`);
                data.results.forEach(result => {
                    if (result.status === 'success') {
                        const analysis = result.analysis;
                        log(`📞 ${result.call_id.substring(0, 8)}: ${analysis.call_result} - ${analysis.main_status || 'غير محدد'}`);
                    } else {
                        log(`❌ ${result.call_id}: ${result.error}`);
                    }
                });
                setTimeout(() => location.reload(), 2000);
            } else {
                log('❌ فشل التحليل: ' + data.error);
            }
            
            btn.disabled = false;
            btn.innerHTML = '<i class="fas fa-play"></i> تحليل المكالمات الجديدة';
        })
        .catch(error => {
            log('❌ خطأ في التحليل: ' + error);
            btn.disabled = false;
            btn.innerHTML = '<i class="fas fa-play"></i> تحليل المكالمات الجديدة';
        });
    }

    function analyzeSingleCall(callId) {
        const btn = event.target;
        btn.disabled = true;
        btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> جاري التحليل...';
        
        log(`🔍 تحليل المكالمة: ${callId.substring(0, 8)}...`);
        
        fetch(`?action=analyze_single&call_id=${callId}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const analysis = data.analysis;
                log(`✅ تم تحليل المكالمة بنجاح:`);
                log(`   📊 نتيجة المكالمة: ${analysis.call_result}`);
                log(`   😊 رضا العميل: ${analysis.customer_satisfaction}`);
                log(`   👤 أداء الوكيل: ${analysis.agent_performance}`);
                log(`   📋 الحالة الرئيسية: ${analysis.main_status || 'غير محددة'}`);
                log(`   📝 الحالة الفرعية: ${analysis.sub_status || 'غير محددة'}`);
                log(`   💡 النقاط المهمة: ${analysis.key_points || 'لا توجد'}`);
                if (analysis.should_call_again) {
                    log(`   📞 يحتاج اتصال تاني - Claude AI ينصح`);
                }
                log(`   📄 الملخص: ${analysis.analysis_summary}`);
                setTimeout(() => location.reload(), 1000);
            } else {
                log(`❌ Claude AI فشل التحليل: ${data.error}`);
            }
            
            btn.disabled = false;
            btn.innerHTML = '<i class="fas fa-play"></i> تحليل الآن';
        })
        .catch(error => {
            log(`❌ خطأ في Claude API: ${error}`);
            btn.disabled = false;
            btn.innerHTML = '<i class="fas fa-play"></i> تحليل الآن';
        });
    }

    function getAnalysis(callId) {
        log(`📖 عرض تحليل المكالمة: ${callId.substring(0, 8)}...`);
        
        fetch(`?action=get_analysis&call_id=${callId}`)
        .then(response => response.json())
        .then(data => {
            if (data.success && data.analysis) {
                const analysis = data.analysis;
                log(`📞 تفاصيل المكالمة ${callId.substring(0, 8)}:`);
                log(`   🎯 النتيجة: ${analysis.ai_analysis.call_result}`);
                log(`   😊 الرضا: ${analysis.ai_analysis.customer_satisfaction}`);
                log(`   📋 الحالة: ${analysis.company_status.main_status || 'غير محدد'}`);
                log(`   📅 تاريخ المتابعة: ${analysis.ai_analysis.follow_up_date || 'غير محدد'}`);
                log(`   📝 الملخص: ${analysis.ai_analysis.analysis_summary}`);
            } else {
                log(`❌ لم يتم العثور على تحليل للمكالمة`);
            }
        })
        .catch(error => {
            log(`❌ خطأ في استرجاع التحليل: ${error}`);
        });
    }

    // Auto-refresh stats every 5 minutes
    setInterval(() => {
        log('🔄 تحديث تلقائي للإحصائيات...');
        loadOverview();
    }, 300000);
    </script>
</body>
</html>
        <?php
    }
}

// Auto-run analysis if called directly or handle web requests
if (basename(__FILE__) == basename($_SERVER['SCRIPT_NAME'])) {
    try {
        $analyzer = new CallAnalyzer();
        
        // Check if this is a web request or API call
        if (!empty($_GET) || !empty($_POST)) {
            $analyzer->handleWebRequest();
        } else {
            // Command line execution - run batch analysis
            echo "🚀 بدء تحليل المكالمات الجديدة...\n";
            $results = $analyzer->analyzeNewCalls();
            
            // Format results for command line output
            $successful = 0;
            $failed = 0;
            
            foreach ($results as $result) {
                if ($result['status'] === 'success') {
                    $successful++;
                    $analysis = $result['analysis'];
                    echo "✅ {$result['call_id']}: {$analysis['call_result']} - {$analysis['main_status']}\n";
                    if ($analysis['follow_up_date']) {
                        echo "   📅 متابعة في: {$analysis['follow_up_date']}\n";
                    }
                } else {
                    $failed++;
                    echo "❌ {$result['call_id']}: {$result['error']}\n";
                }
            }
            
            echo "\n📊 النتائج النهائية:\n";
            echo "✅ تم تحليل: {$successful} مكالمة\n";
            echo "❌ فشل في: {$failed} مكالمة\n";
            echo "🎯 إجمالي: " . count($results) . " مكالمة\n";
        }
        
    } catch (Exception $e) {
        if (!empty($_GET) || !empty($_POST)) {
            header('Content-Type: application/json; charset=utf-8');
            echo json_encode(['success' => false, 'error' => $e->getMessage()], JSON_UNESCAPED_UNICODE);
        } else {
            echo "❌ خطأ: " . $e->getMessage() . "\n";
        }
    }
}
?>