<?php
require_once __DIR__ . '/vapi_config.php';

class CallAnalyzer {
    private $pdo;
    private $openai_api_key;
    
    public function __construct() {
        $this->pdo = new PDO(DB_DSN, DB_USER, DB_PASS, [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4"
        ]);
        
        // Add your ChatGPT API key here
        $this->openai_api_key = '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';
    }
    
    public function analyzeCall($call_id) {
        // Get call details
        $call = $this->getCallDetails($call_id);
        if (!$call) {
            return ['error' => 'Call not found'];
        }
        
        // Get transcript
        $transcript = $this->getTranscript($call_id);
        if (empty($transcript)) {
            return ['error' => 'No transcript available'];
        }
        
        // Format transcript for analysis
        $formatted_transcript = $this->formatTranscript($transcript);
        
        // Rate limiting - wait 1 second between API calls
        sleep(1);
        
        // Analyze with ChatGPT AI
        $analysis = $this->performChatGPTAnalysis($formatted_transcript, $call);
        
        if (isset($analysis['error'])) {
            return $analysis;
        }
        
        // Save analysis to database
        $this->saveAnalysis($call_id, $analysis);
        
        return $analysis;
    }
    
    private function getCallDetails($call_id) {
        $stmt = $this->pdo->prepare("SELECT * FROM vapi_call_logs WHERE call_id = ?");
        $stmt->execute([$call_id]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    private function getTranscript($call_id) {
        $stmt = $this->pdo->prepare("
            SELECT role, message, timestamp 
            FROM vapi_call_transcripts 
            WHERE call_id = ? 
            ORDER BY sequence_number ASC
        ");
        $stmt->execute([$call_id]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    private function formatTranscript($transcript) {
        $formatted = "";
        foreach ($transcript as $entry) {
            $role = ucfirst($entry['role']);
            $message = trim($entry['message']);
            if (!empty($message)) {
                $formatted .= "$role: $message\n";
            }
        }
        return $formatted;
    }
    
    private function performChatGPTAnalysis($transcript, $call_details) {
        $prompt = $this->buildAnalysisPrompt($transcript, $call_details);
        
        $response = $this->callChatGPT($prompt);
        
        if (isset($response['error'])) {
            return $response;
        }
        
        // Parse ChatGPT AI response
        return $this->parseChatGPTResponse($response['choices'][0]['message']['content']);
    }
    
    private function buildAnalysisPrompt($transcript, $call_details) {
        return "حلل محادثة خدمة العملاء دي وقولي رأيك:

تفاصيل المكالمة:
- المدة: " . ($call_details['duration_seconds'] ?? 0) . " ثانية
- سبب انتهاء المكالمة: " . ($call_details['ended_reason'] ?? 'مش محدد') . "
- المساعد: " . ($call_details['assistant_name'] ?? 'مش محدد') . "

نص المحادثة:
$transcript

عايزك تحلل المكالمة دي وترد بـ JSON بالشكل ده:
{
    \"sentiment_score\": (رقم من -1.0 إلى 1.0، حيث -1 = سلبي جداً، 0 = عادي، +1 = إيجابي جداً),
    \"key_topics\": [\"موضوع 1\", \"موضوع 2\", \"موضوع 3\"],
    \"action_items\": [\"مطلوب عمل كذا\", \"لازم نتابع كذا\"],
    \"customer_satisfaction\": (تقييم من 1-10),
    \"analysis_notes\": \"تحليل مفصل لجودة المكالمة والمشاكل والحلول المقترحة\",
    \"call_resolution\": \"اتحل\" | \"اتحل جزئياً\" | \"ماتحلش\",
    \"agent_performance\": \"ممتاز\" | \"كويس\" | \"متوسط\" | \"محتاج تحسين\",
    \"customer_emotion\": \"راضي\" | \"عادي\" | \"متضايق\" | \"زعلان\",
    \"follow_up_required\": true | false,
    \"call_category\": \"دعم فني\" | \"مبيعات\" | \"شكوى\" | \"استفسار\" | \"أخرى\"
}

ركز على النصايح العملية لتحسين الخدمة. ارد بـ JSON فقط، بلاش أي كلام تاني.";
    }

    private function callChatGPT($prompt) {
        $data = [
            'model' => 'gpt-4',
            'messages' => [
                [
                    'role' => 'system',
                    'content' => 'أنت محلل مكالمات خدمة عملاء محترف. تحلل المكالمات وتعطي تقييمات دقيقة ومفصلة باللغة العربية.'
                ],
                [
                    'role' => 'user',
                    'content' => $prompt
                ]
            ],
            'max_tokens' => 800,
            'temperature' => 0.3
        ];
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, 'https://api.openai.com/v1/chat/completions');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Authorization: Bearer ' . $this->openai_api_key
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlError = curl_error($ch);
        curl_close($ch);
        
        if ($curlError) {
            error_log("ChatGPT CURL Error: " . $curlError);
            return ['error' => 'Network error: ' . $curlError];
        }
        
        if ($httpCode !== 200) {
            error_log("ChatGPT API Error: HTTP $httpCode - Response: $response");
            return ['error' => 'ChatGPT API error: HTTP ' . $httpCode];
        }
        
        return json_decode($response, true);
    }
    
    private function parseChatGPTResponse($response) {
        // Clean up the response (remove any markdown formatting)
        $response = trim($response);
        $response = preg_replace('/^```json\s*/', '', $response);
        $response = preg_replace('/\s*```$/', '', $response);
        
        $analysis = json_decode($response, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            error_log("JSON Parse Error: " . json_last_error_msg() . " | Response: " . $response);
            return ['error' => 'Failed to parse ChatGPT response: ' . json_last_error_msg()];
        }
        
        // Validate and sanitize the analysis
        return $this->validateAnalysis($analysis);
    }
    
    private function validateAnalysis($analysis) {
        // Set defaults for missing fields
        $validated = [
            'sentiment_score' => $this->validateFloat($analysis['sentiment_score'] ?? 0, -1, 1),
            'customer_satisfaction' => $this->validateInt($analysis['customer_satisfaction'] ?? 5, 1, 10),
            'key_topics' => $this->validateArray($analysis['key_topics'] ?? []),
            'action_items' => $this->validateArray($analysis['action_items'] ?? []),
            'analysis_notes' => $this->validateString($analysis['analysis_notes'] ?? ''),
            'call_resolution' => $this->validateString($analysis['call_resolution'] ?? 'unresolved'),
            'agent_performance' => $this->validateString($analysis['agent_performance'] ?? 'average'),
            'customer_emotion' => $this->validateString($analysis['customer_emotion'] ?? 'neutral'),
            'follow_up_required' => $analysis['follow_up_required'] ?? false,
            'call_category' => $this->validateString($analysis['call_category'] ?? 'other')
        ];
        
        return $validated;
    }
    
    private function validateFloat($value, $min, $max) {
        $float = floatval($value);
        return max($min, min($max, $float));
    }
    
    private function validateInt($value, $min, $max) {
        $int = intval($value);
        return max($min, min($max, $int));
    }
    
    private function validateArray($value) {
        return is_array($value) ? $value : [];
    }
    
    private function validateString($value) {
        return is_string($value) ? trim($value) : '';
    }
    
    private function saveAnalysis($call_id, $analysis) {
        if (isset($analysis['error'])) {
            return false;
        }
        
        $sql = "INSERT INTO vapi_call_analysis 
                (call_id, sentiment_score, key_topics, action_items, customer_satisfaction, analysis_notes)
                VALUES (?, ?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE
                    sentiment_score = VALUES(sentiment_score),
                    key_topics = VALUES(key_topics),
                    action_items = VALUES(action_items),
                    customer_satisfaction = VALUES(customer_satisfaction),
                    analysis_notes = VALUES(analysis_notes)";
        
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([
            $call_id,
            $analysis['sentiment_score'],
            is_array($analysis['key_topics']) ? implode(', ', $analysis['key_topics']) : '',
            is_array($analysis['action_items']) ? implode(', ', $analysis['action_items']) : '',
            $analysis['customer_satisfaction'],
            $analysis['analysis_notes']
        ]);
        
        return true;
    }
    
    public function analyzeRecentCalls($limit = 10) {
        // Get calls that haven't been analyzed yet
        $stmt = $this->pdo->prepare("
            SELECT cl.call_id, cl.assistant_name, cl.start_time
            FROM vapi_call_logs cl
            LEFT JOIN vapi_call_analysis ca ON cl.call_id = ca.call_id
            WHERE ca.call_id IS NULL 
            AND cl.duration_seconds IS NOT NULL
            AND cl.duration_seconds > 0
            ORDER BY cl.created_at DESC
            LIMIT ?
        ");
        $stmt->execute([$limit]);
        $calls = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $results = [];
        foreach ($calls as $call) {
            echo "Analyzing call with ChatGPT AI: " . substr($call['call_id'], 0, 8) . "...\n";
            $results[$call['call_id']] = $this->analyzeCall($call['call_id']);
            
            // Add delay between calls to avoid rate limits
            if (count($calls) > 1) {
                sleep(2);
            }
        }
        
        return $results;
    }
    
    public function getAnalyticsOverview() {
        $stmt = $this->pdo->query("
            SELECT 
                COUNT(*) as total_analyzed_calls,
                AVG(sentiment_score) as avg_sentiment,
                AVG(customer_satisfaction) as avg_satisfaction,
                COUNT(CASE WHEN sentiment_score >= 0.5 THEN 1 END) as positive_calls,
                COUNT(CASE WHEN sentiment_score <= -0.5 THEN 1 END) as negative_calls
            FROM vapi_call_analysis
        ");
        
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
}

// Handle AJAX requests
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    $analyzer = new CallAnalyzer();
    
    header('Content-Type: application/json');
    
    try {
        if ($_POST['action'] === 'analyze_call' && isset($_POST['call_id'])) {
            $result = $analyzer->analyzeCall($_POST['call_id']);
            echo json_encode($result);
            exit;
        }
        
        if ($_POST['action'] === 'analyze_recent') {
            $limit = isset($_POST['limit']) ? (int)$_POST['limit'] : 10;
            $results = $analyzer->analyzeRecentCalls($limit);
            echo json_encode($results);
            exit;
        }
        
        if ($_POST['action'] === 'get_overview') {
            $overview = $analyzer->getAnalyticsOverview();
            echo json_encode($overview);
            exit;
        }
    } catch (Exception $e) {
        echo json_encode(['error' => $e->getMessage()]);
        exit;
    }
}

// Web interface
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <title>محلل المكالمات بـ ChatGPT AI</title>
    <meta name="viewport" content="width=device-width,initial-scale=1">
    <meta charset="UTF-8">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body { 
            font-family: 'Segoe UI', 'Cairo', Tahoma, Arial, sans-serif; 
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            direction: rtl;
            color: #333;
        }
        
        .header {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            padding: 20px 0;
            box-shadow: 0 2px 20px rgba(0,0,0,0.1);
            margin-bottom: 30px;
        }
        
        .container { 
            max-width: 1200px; 
            margin: 0 auto; 
            padding: 0 20px;
        }
        
        h1 {
            text-align: center;
            color: #2d3748;
            font-size: 2.2rem;
            font-weight: 700;
            margin-bottom: 10px;
        }
        
        .subtitle {
            text-align: center;
            color: #718096;
            font-size: 1.1rem;
            margin-bottom: 0;
        }
        
        .chatgpt-badge {
            display: inline-block;
            background: linear-gradient(135deg, #10a37f 0%, #1a7f64 100%);
            color: white;
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            margin-left: 10px;
        }
        
        .card { 
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            border: none;
            border-radius: 20px; 
            padding: 30px; 
            margin: 25px 0; 
            box-shadow: 0 8px 32px rgba(0,0,0,0.1);
            transition: all 0.3s ease;
        }
        
        .card:hover {
            transform: translateY(-5px);
            box-shadow: 0 15px 40px rgba(0,0,0,0.15);
        }
        
        .card h2 { 
            color: #2d3748;
            font-size: 1.4rem;
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .card h2 i {
            color: #10a37f;
            font-size: 1.2rem;
        }
        
        .btn { 
            background: linear-gradient(135deg, #10a37f 0%, #1a7f64 100%);
            color: white; 
            padding: 14px 28px; 
            border: none; 
            border-radius: 12px; 
            cursor: pointer; 
            font-size: 15px;
            font-weight: 600;
            margin: 8px 0 8px 8px;
            transition: all 0.3s ease;
            box-shadow: 0 4px 15px rgba(0,0,0,0.2);
        }
        
        .btn:hover { 
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(0,0,0,0.25);
        }
        
        .btn:disabled { 
            background: #cbd5e0; 
            cursor: not-allowed;
            transform: none;
            box-shadow: none;
        }
        
        .btn-success { 
            background: linear-gradient(135deg, #48bb78 0%, #38a169 100%);
        }
        
        .btn-success:hover { 
            background: linear-gradient(135deg, #38a169 0%, #2f855a 100%);
        }
        
        .form-group { 
            margin-bottom: 20px; 
        }
        
        .form-group label { 
            display: block; 
            margin-bottom: 8px; 
            font-weight: 600;
            color: #4a5568;
            font-size: 15px;
        }
        
        .form-group input { 
            width: 100%; 
            max-width: 350px; 
            padding: 14px 18px; 
            border: 2px solid #e2e8f0; 
            border-radius: 12px;
            font-size: 15px;
            transition: all 0.3s ease;
            background: #f7fafc;
        }
        
        .form-group input:focus {
            outline: none;
            border-color: #10a37f;
            background: white;
            box-shadow: 0 0 0 3px rgba(16, 163, 127, 0.1);
        }
        
        .overview-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin: 25px 0;
        }
        
        .metric {
            background: linear-gradient(135deg, #f7fafc 0%, #edf2f7 100%);
            padding: 25px;
            border-radius: 16px;
            text-align: center;
            border: 1px solid #e2e8f0;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }
        
        .metric:before {
            content: '';
            position: absolute;
            top: 0;
            right: 0;
            width: 100%;
            height: 4px;
            background: linear-gradient(90deg, #10a37f 0%, #1a7f64 100%);
        }
        
        .metric:hover {
            transform: translateY(-3px);
            box-shadow: 0 10px 25px rgba(0,0,0,0.1);
        }
        
        .metric-value {
            font-size: 2.2rem;
            font-weight: 700;
            color: #2d3748;
            margin-bottom: 8px;
        }
        
        .metric-label {
            font-size: 14px;
            color: #718096;
            font-weight: 500;
        }
        
        #output {
            background: #1a202c;
            color: #e2e8f0;
            padding: 20px;
            border-radius: 12px;
            font-family: 'Courier New', monospace;
            font-size: 14px;
            min-height: 120px;
            overflow-y: auto;
            white-space: pre-wrap;
            direction: rtl;
            border: 1px solid #2d3748;
            line-height: 1.6;
        }
        
        .description {
            color: #718096;
            font-size: 15px;
            margin-bottom: 20px;
            line-height: 1.6;
        }
        
        .loading {
            display: inline-block;
            width: 16px;
            height: 16px;
            border: 2px solid #cbd5e0;
            border-radius: 50%;
            border-top: 2px solid #10a37f;
            animation: spin 1s linear infinite;
            margin-left: 8px;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        .success-message {
            color: #38a169;
            font-weight: 600;
        }
        
        .error-message {
            color: #e53e3e;
            font-weight: 600;
        }
        
        .alert {
            padding: 16px;
            border-radius: 12px;
            margin: 20px 0;
            border: 1px solid;
        }
        
        .alert-info {
            background: #e6fffa;
            color: #234e52;
            border-color: #81e6d9;
        }
        
        @media (max-width: 768px) {
            .container {
                padding: 0 15px;
            }
            
            .card {
                padding: 20px;
                margin: 15px 0;
            }
            
            h1 {
                font-size: 1.8rem;
            }
            
            .overview-grid {
                grid-template-columns: 1fr;
                gap: 15px;
            }
        }
    </style>
</head>
<body>
    <div class="header">
        <div class="container">
            <h1>
                <span class="chatgpt-badge">ChatGPT AI</span>
                <i class="fas fa-brain"></i> محلل المكالمات الذكي
            </h1>
            <p class="subtitle">اكتشف ما يحدث حقاً في مكالماتك مع ذكاء ChatGPT</p>
        </div>
    </div>

    <div class="container">
        <!-- New Feature Alert -->
        <div class="alert alert-info">
            <i class="fas fa-star"></i>
            <strong>جديد!</strong> النظام الآن يعمل بـ ChatGPT AI GPT-4 لتحليل أكثر دقة وذكاء!
        </div>

        <!-- Analytics Overview -->
        <div class="card">
            <h2><i class="fas fa-chart-bar"></i> إحصائيات سريعة</h2>
            <button onclick="loadOverview()" class="btn">حدّث الإحصائيات</button>
            <div id="overview" class="overview-grid">
                <div class="metric">
                    <div class="metric-value" id="totalCalls">-</div>
                    <div class="metric-label">مكالمة تم تحليلها</div>
                </div>
                <div class="metric">
                    <div class="metric-value" id="avgSentiment">-</div>
                    <div class="metric-label">مستوى الرضا العام</div>
                </div>
                <div class="metric">
                    <div class="metric-value" id="avgSatisfaction">-</div>
                    <div class="metric-label">تقييم العملاء</div>
                </div>
                <div class="metric">
                    <div class="metric-value" id="positiveCalls">-</div>
                    <div class="metric-label">مكالمة ناجحة</div>
                </div>
            </div>
        </div>

        <!-- Batch Analysis -->
        <div class="card">
            <h2><i class="fas fa-tasks"></i> حلل مكالمات كتير مرة واحدة</h2>
            <p class="description">ChatGPT AI يحلل لك آخر مكالمات ماحللتهاش عشان تشوف إيه اللي بيحصل</p>
            <div class="form-group">
                <label>كام مكالمة عاوز تحللها؟</label>
                <input type="number" id="batchLimit" value="5" min="1" max="50">
            </div>
            <button onclick="analyzeBatch()" class="btn btn-success" id="batchBtn">
                <i class="fas fa-play"></i> يلا نبدأ التحليل بـ ChatGPT
            </button>
        </div>
        
        <!-- Single Call Analysis -->
        <div class="card">
            <h2><i class="fas fa-search"></i> حلل مكالمة واحدة</h2>
            <p class="description">عندك مكالمة معينة عاوز تفهم إيه اللي حصل فيها؟ ChatGPT AI هيساعدك</p>
            <div class="form-group">
                <label>رقم المكالمة:</label>
                <input type="text" id="callId" placeholder="اكتب رقم المكالمة هنا">
            </div>
            <button onclick="analyzeSingle()" class="btn" id="singleBtn">
                <i class="fas fa-microscope"></i> حلل بـ ChatGPT AI
            </button>
        </div>

        <!-- Output -->
        <div class="card">
            <h2><i class="fas fa-terminal"></i> نتايج ChatGPT AI</h2>
            <div id="output">🤖 ChatGPT AI مستني أمرك... اختار إيه اللي عاوز تحلله

✨ ChatGPT GPT-4 جاهز لتحليل ذكي ومفصل لمكالماتك</div>
        </div>
    </div>

    <script>
    function log(message) {
        const output = document.getElementById('output');
        const timestamp = new Date().toLocaleTimeString();
        output.textContent += `[${timestamp}] ${message}\n`;
        output.scrollTop = output.scrollHeight;
    }

    function loadOverview() {
        fetch('', {
            method: 'POST',
            headers: {'Content-Type': 'application/x-www-form-urlencoded'},
            body: 'action=get_overview'
        })
        .then(response => response.json())
        .then(data => {
            if (data.error) {
                log('Overview error: ' + data.error);
                return;
            }
            
            document.getElementById('totalCalls').textContent = data.total_analyzed_calls || '0';
            document.getElementById('avgSentiment').textContent = data.avg_sentiment ? parseFloat(data.avg_sentiment).toFixed(2) : '0.00';
            document.getElementById('avgSatisfaction').textContent = data.avg_satisfaction ? parseFloat(data.avg_satisfaction).toFixed(1) : '0.0';
            document.getElementById('positiveCalls').textContent = data.positive_calls || '0';
        })
        .catch(error => log('Overview failed: ' + error));
    }

    function analyzeBatch() {
        const btn = document.getElementById('batchBtn');
        const limit = document.getElementById('batchLimit').value;
        
        btn.disabled = true;
        btn.textContent = 'ChatGPT AI يحلل...';
        log(`🤖 ChatGPT AI بدأ تحليل ${limit} مكالمة بذكاء عالي...`);
        
        fetch('', {
            method: 'POST',
            headers: {'Content-Type': 'application/x-www-form-urlencoded'},
            body: `action=analyze_recent&limit=${limit}`
        })
        .then(response => response.json())
        .then(data => {
            if (data.error) {
                log('❌ ChatGPT AI error: ' + data.error);
            } else {
                const successCount = Object.values(data).filter(result => !result.error).length;
                const errorCount = Object.values(data).length - successCount;
                log(`✅ ChatGPT AI خلص التحليل: ${successCount} نجح، ${errorCount} فشل`);
                
                for (const [callId, result] of Object.entries(data)) {
                    const shortId = callId.substring(0, 8) + '...';
                    if (result.error) {
                        log(`❌ ${shortId}: ${result.error}`);
                    } else {
                        log(`🎯 ${shortId}: المشاعر ${result.sentiment_score?.toFixed(2)}، الرضا ${result.customer_satisfaction}/10`);
                    }
                }
            }
            
            btn.disabled = false;
            btn.textContent = 'يلا نبدأ التحليل بـ ChatGPT';
            loadOverview();
        })
        .catch(error => {
            log('❌ خطأ في ChatGPT API: ' + error);
            btn.disabled = false;
            btn.textContent = 'يلا نبدأ التحليل بـ ChatGPT';
        });
    }

    function analyzeSingle() {
        const btn = document.getElementById('singleBtn');
        const callId = document.getElementById('callId').value.trim();
        
        if (!callId) {
            alert('لازم تكتب رقم المكالمة الأول');
            return;
        }
        
        btn.disabled = true;
        btn.textContent = 'ChatGPT AI يحلل...';
        log(`🔍 ChatGPT AI بيحلل المكالمة: ${callId.substring(0, 8)}...`);
        
        fetch('', {
            method: 'POST',
            headers: {'Content-Type': 'application/x-www-form-urlencoded'},
            body: `action=analyze_call&call_id=${callId}`
        })
        .then(response => response.json())
        .then(data => {
            if (data.error) {
                log('❌ ChatGPT AI فشل التحليل: ' + data.error);
            } else {
                log(`✅ ChatGPT AI خلص التحليل:`);
                log(`   📊 المشاعر: ${data.sentiment_score?.toFixed(2)} (من -1 إلى +1)`);
                log(`   🎯 الرضا: ${data.customer_satisfaction}/10`);
                log(`   📋 المواضيع: ${data.key_topics?.join('، ') || 'مافيش'}`);
                log(`   💡 الملاحظات: ${data.analysis_notes?.substring(0, 100)}...`);
                log(`   🤖 ChatGPT AI يقول: ${data.agent_performance} أداء الوكيل`);
            }
            
            btn.disabled = false;
            btn.textContent = 'حلل بـ ChatGPT AI';
            loadOverview();
        })
        .catch(error => {
            log('❌ خطأ في ChatGPT API: ' + error);
            btn.disabled = false;
            btn.textContent = 'حلل بـ ChatGPT AI';
        });
    }

    // Load overview on page load
    loadOverview();
    </script>
</body>
</html>